/* The following code example is described in the book "Introduction
 * to Geometric Computing" by Sherif Ghali, Springer-Verlag, 2008.
 *
 * Copyright (C) 2008 Sherif Ghali. This code may be freely copied,
 * modified, or republished electronically or in print provided that
 * this copyright notice appears in all copies. This software is
 * provided "as is" without express or implied warranty; not even for
 * merchantability or fitness for a particular purpose.
 */

#ifndef POSTSCRIPT_H
#define POSTSCRIPT_H

#include <algorithm>
#include <iterator>

#include <fstream>
#include <list>
#include <iostream>
#include <iomanip>
#include <cstdlib>
#include <string>

using std::endl;

#include "../misc/conversions.h" // to_double

#include "../geometry_E2/point_e2.h"
#include "../geometry_E2/line_e2.h"
#include "../geometry_E2/segment_e2.h"
#include "../geometry_E2/triangle_e2.h"
#include "../geometry_E2/bbox_e2.h"

#include "../geometry_S1/segment_s1.h"

typedef Point_E2<double>   PS_Point_E2d;

template<class T>
class Postscript;

template<class T>
class Postscript
{
    // Letter: 8.5"x11"
    // A4:     297mmx210mm

    static const double Xoffset;
    static const double Yoffset;
    static const double PS_WIDTH;
    static const double PS_HEIGHT;

private:
    std::string   filename;
    std::ofstream  outfile;

    Bbox_E2<T>     world_bbox;	// world coordinates
    Bbox_E2<double>     psbox;	// PS coordinates

    double nodeDiameter;
    double graylevelfill;
    double graylevelstroke;

    double line_width;

    void write_ps_header();

    PS_Point_E2d world_to_ps_transform(const Point_E2<T>& pin);
    T world_to_ps_scale(const T& distance);

public:
    Postscript(const std::string& filename,
	       const Bbox_E2<T>& _world_bbox
	       /*, bool isometry = true*/);

    virtual ~Postscript();

    void moveto(const Point_E2<T>& p);
    void lineto(const Point_E2<T>& p);

    void draw(const Bbox_E2<T>& B);
    void draw_square(const Bbox_E2<T>& B);
    void draw_bounding_box();

    void draw(const Point_E2<T>& P);

private:
    void draw_circle_or_disk(const Point_E2<T>& p, const T& radius);
public:
    void draw_circle(const Point_E2<T>& p, const T& radius);
    void draw_disk(const Point_E2<T>& p, const T& radius);

    void draw_small_circle(const Point_E2<T>& p, double dfraction=100.0);
    void draw_small_disk(const Point_E2<T>& p, double dfraction=100.0);

private:
    void draw_box_or_square(const Point_E2<T>& p, const T& radius);
public:
    void draw_box(const Point_E2<T>& p, const T& radius);
    void draw_square(const Point_E2<T>& p, const T& radius);

    void draw(const Segment_S1<T>& S,
	      const Point_E2<T>& P,	      
	      const T& radius);

    void draw(const Segment_E2<T>& seg);
    void draw(const Triangle_E2<T>& T, bool fill_triangle=false);

    void draw_arrow(const Segment_E2<T>& s);

    void draw_axes();

    void draw(const std::string& s, const Point_E2<T>& p, const T& radius);

    void draw_grid();

    void set_gray_fill(double graylevel) {
	graylevelfill = graylevel;
	outfile << graylevelfill << " G" << endl;
    }
    void set_gray_stroke(double graylevel) {
	graylevelstroke = graylevel;
	outfile << graylevelstroke << " G" << endl;
    }
    void set_gray(double graylevel) // 0.0: black; 1.0: white
    {
	outfile << graylevel << " G" << endl;
    }

    void set_line_width(double lw) {
	line_width = lw;
	outfile << line_width << " W\n\n";
    }

    void set_line_cap(int lc) {
	outfile << lc << " LC\n\n";
    }
    void set_line_join(int lj) {
	outfile << lj << " LJ\n\n";
    }

    void newpath() { outfile << "N\n"; }
    void closepath() { outfile << "C\n"; }
    void stroke() { outfile << "S\n"; }
    void fill() { outfile << "F\n"; }

    void close() {
	outfile << "showpage\n";
	outfile.close();
    }

};

template<typename T>
const double
Postscript<T>::Xoffset = 72; // leave 1" on either

template<typename T>
const double
Postscript<T>::Yoffset = 72; // letter-sized or A4 paper

template<typename T>
const double
Postscript<T>::PS_WIDTH = 454;	// width = (210 - 25 - 25) / 25.4 * 72 ~= 454

template<typename T>
const double
Postscript<T>::PS_HEIGHT = 648; // height = 9 * 72 = 648


template<class T>
Postscript<T>::Postscript(const std::string& filename,
			  const Bbox_E2<T>& _world_bbox
			  /*, bool isometry = true*/)
    : filename(filename), 
      world_bbox(_world_bbox),
      graylevelfill(0.5), graylevelstroke(0.0)
{
    outfile.open(filename.c_str());
    if( !outfile ) {
	std::cerr<<"error: cannot open file "<<filename<<" for postscript output\n";
	exit(1);
    }

    write_ps_header();
}

template<class T>
Postscript<T>::~Postscript() {
    close();
    outfile.close();
}

template<class T>
void
Postscript<T>::write_ps_header()
{
    double boxWidth  = to_double(world_bbox.UR().x() - world_bbox.LL().x());
    double boxHeight = to_double(world_bbox.UR().y() - world_bbox.LL().y());

    if(boxWidth / boxHeight > PS_WIDTH / PS_HEIGHT)
	psbox = Bbox_E2<double>(PS_Point_E2d(Xoffset,Yoffset),
				PS_Point_E2d(Xoffset+PS_WIDTH,
					     Yoffset+boxHeight/boxWidth * PS_WIDTH));
    else
	psbox = Bbox_E2<double>(PS_Point_E2d(Xoffset,Yoffset),
				PS_Point_E2d(Xoffset+boxWidth/boxHeight * PS_HEIGHT,
					     Yoffset+PS_HEIGHT));

    outfile << "%!PS-Adobe-2.0\n"
	    << "%%Creator: Postscript\n"
	    << "%%Title: " << filename << endl
	    << "%%BoundingBox: "
	    << psbox.LL().x() << " " << psbox.LL().y() << " "
	    << psbox.UR().x() << " " << psbox.UR().y() << endl
	    << "%%EndComments\n\n";

    outfile << "% Macro definitions:\n"
	    << "/N {newpath} def \n"
	    << "/M {moveto} def\n"
	    << "/L {lineto} def\n"
	    << "/C {closepath} def\n"
	    << "/S {stroke} def\n"
	    << "/F {fill} def\n"
	    << "/W {setlinewidth} def\n"
	    << "/G {setgray} def\n"
	    << "/LC {setlinecap} def\n"
	    << "/LJ {setlinejoin} def\n"
	    << "% Other setup\n"
	    << "1 setlinecap\n"   // 0: butt caps,    1: round caps,  2: line caps
	    << "1 setlinejoin\n"  // 0: mitter joins, 1: round joins, 2: bevel joins
	    << "/Times-Bold findfont\n"
	    << "15 scalefont\n"
	    << "setfont\n"
	    << "%%EndProlog\n\n"
	    << std::setprecision(6);

    // set_line_width(0.01);

    nodeDiameter = PS_WIDTH/180.0;
    outfile.flush();
}

template<class T>
void
Postscript<T>::draw_bounding_box()
{
    outfile << graylevelstroke << " G" << endl;

    outfile << "N" << endl
	    << psbox.LL().x() << " " << psbox.LL().y() << " M" << endl
	    << psbox.LL().x() << " " << psbox.UR().y() << " L" << endl
	    << psbox.UR().x() << " " << psbox.UR().y() << " L" << endl
	    << psbox.UR().x() << " " << psbox.LL().y() << " L" << endl
	    << "C S" << endl;
}

template<class T>
void
Postscript<T>::draw_small_circle(const Point_E2<T>& p, double dfraction)
{
    double worldpsBoxWidth  = to_double(world_bbox.UR().x() - world_bbox.LL().x());
    draw_circle( p, worldpsBoxWidth / dfraction );
}

template<class T>
void
Postscript<T>::draw_small_disk(const Point_E2<T>& p, double dfraction)
{
    double worldpsBoxWidth  = to_double(world_bbox.UR().x() - world_bbox.LL().x());
    draw_disk( p, worldpsBoxWidth / dfraction );
}

template<class T>
void
Postscript<T>::draw(const Point_E2<T>& p)
{
    draw_small_disk(p);
}

template<class T>
void
Postscript<T>::draw_circle_or_disk(const Point_E2<T>& p, const T& radius)
{
    PS_Point_E2d pd = world_to_ps_transform(p);
    T rd = world_to_ps_scale(radius);
    outfile << pd.x() << " " << pd.y() << " M" << endl;
    outfile << "N " << pd.x() << " " << pd.y() << " " << rd
	    << "  0 360 arc" << endl;
}

template<class T>
void
Postscript<T>::draw_circle(const Point_E2<T>& p, const T& radius)
{
    draw_circle_or_disk(p, radius);
    stroke();
}

template<class T>
void
Postscript<T>::draw_disk(const Point_E2<T>& p, const T& radius)
{
    draw_circle_or_disk(p, radius);
    fill();
}

template<class T>
void
Postscript<T>::draw_box_or_square(const Point_E2<T>& p, const T& radius)
{
    PS_Point_E2d pd = world_to_ps_transform(p);
    T rd = world_to_ps_scale(radius);
    T rd10 = rd/5.0;

    outfile << rd10/2.0 << " W\n";

    PS_Point_E2d LLbegin(pd.x() - 0.8 * rd , pd.y() - rd);

    PS_Point_E2d LL(pd.x() - rd , pd.y() - rd);
    PS_Point_E2d LR(pd.x() + rd , pd.y() - rd);
    PS_Point_E2d UR(pd.x() + rd , pd.y() + rd);
    PS_Point_E2d UL(pd.x() - rd , pd.y() + rd);

    newpath();
    outfile << LLbegin.x() << " " << LLbegin.y() << " M" << endl;
    outfile << LR.x() << " " << LR.y() << " " << UR.x() << " " << UR.y() << " " << rd10 << " arct" << endl;
    outfile << UR.x() << " " << UR.y() << " " << UL.x() << " " << UL.y() << " " << rd10 << " arct" << endl;
    outfile << UL.x() << " " << UL.y() << " " << LL.x() << " " << LL.y() << " " << rd10 << " arct" << endl;
    outfile << LL.x() << " " << LL.y() << " " << LR.x() << " " << LR.y() << " " << rd10 << " arct" << endl;
}

template<class T>
void
Postscript<T>::draw_box(const Point_E2<T>& p, const T& radius)
{
    draw_box_or_square(p, radius);
    stroke();
}

template<class T>
void
Postscript<T>::draw_square(const Point_E2<T>& p, const T& radius)
{
    draw_box_or_square(p, radius);
    fill();
}

template<class T>
void
Postscript<T>::draw(const Segment_S1<T>& S,
		    const Point_E2<T>& P,
		    const T& radius)
{
    PS_Point_E2d pd = world_to_ps_transform(P);
    T rd = world_to_ps_scale(radius);
    double source_angle = 180.0 / M_PI * angle(S.source());
    double target_angle = 180.0 / M_PI * angle(S.target());

    outfile << pd.x() << " " << pd.y() << " "
	    << rd << "  "
	    << source_angle << " " << target_angle
	    << " arc S" << endl;
}

template<class T>
void
Postscript<T>::draw(const Segment_E2<T>& seg)
{
    newpath();
    moveto(seg.source());
    lineto(seg.target());
    stroke();
}

template<class T>
void
Postscript<T>::draw(const Triangle_E2<T>& triangle, bool fill_triangle)
{
    newpath();
    moveto(triangle.P0());
    lineto(triangle.P1());
    lineto(triangle.P2());
    closepath();
    if(fill_triangle)
	fill();
    else
	stroke();
}

template<class T>
void
Postscript<T>::draw_arrow(const Segment_E2<T>& s)
{
    Vector_E2<T> stem = s.target() - s.source();
    stem = stem * T(0.98);
    Segment_E2<T> stem_seg(s.source(), s.source() + stem);
    draw(stem_seg);

    Vector_E2<T> v1 = s.source() - s.target();
    v1 = v1 / T(25.0);
    Vector_E2<T> v2 = v1;    

    v1 = v1 + Vector_E2<T>( v1.y(), -v1.x()) / T(3.0);
    v2 = v2 + Vector_E2<T>(-v2.y(),  v2.x()) / T(3.0);

    Triangle_E2<T> tri(s.target(),
		       Point_E2<T>(s.target() + v1),
		       Point_E2<T>(s.target() + v2));
    draw(tri, true);

//    alternatively:
//    draw(Segment_E2<T>(s.target(), Point_E2<T>(s.target() + v1)));
//    draw(Segment_E2<T>(s.target(), Point_E2<T>(s.target() + v2)));
}

template<class T>
void
Postscript<T>::draw_axes()
{
    Point_E2<T> left(world_bbox.LL().x(), 0);
    Point_E2<T> right(world_bbox.UR().x(), 0);
    draw_arrow( Segment_E2<T>(left, right) );

    Point_E2<T> bottom(0, world_bbox.LL().y());
    Point_E2<T> top(0, world_bbox.UR().y());
    draw_arrow( Segment_E2<T>(bottom, top) );
}

template<class T>
void
Postscript<T>::draw(const std::string& s, const Point_E2<T>& p, const T& radius)
{
    PS_Point_E2d pd = world_to_ps_transform(p);
    T rd = world_to_ps_scale(radius);
    outfile << pd.x() + rd * 1.25 << " " << pd.y() << " M" << endl;
    outfile << "(" << s << ") show" << endl;
}

template<class T>
void
Postscript<T>::draw_grid() // just the integer grid
{
    for(int x = int(world_bbox.LL().x())-1;
	x <= int(world_bbox.UR().x())+1; ++x)
    {
	Point_E2<T> bottom(x, world_bbox.LL().y());
	Point_E2<T> top(x, world_bbox.UR().y());
	draw( Segment_E2<T>(bottom, top) );
    }							

    for(int y = int(world_bbox.LL().y())-1;
	y <= int(world_bbox.UR().y())+1; ++y)
    {
	Point_E2<T> left(world_bbox.LL().x(), y);
	Point_E2<T> right(world_bbox.UR().x(), y);
	draw( Segment_E2<T>(left, right) );
    }
}

template<class T>
void
Postscript<T>::draw(const Bbox_E2<T>& B)
{
    newpath();
    moveto(B.LL());
    lineto(Point_E2<T>(B.UR().x(), B.LL().y()));
    lineto(B.UR());
    lineto(Point_E2<T>(B.LL().x(), B.UR().y()));
    closepath();
    stroke();
}

template<class T>
void
Postscript<T>::draw_square(const Bbox_E2<T>& B)
{
    newpath();
    moveto(B.LL());
    lineto(Point_E2<T>(B.UR().x(), B.LL().y()));
    lineto(B.UR());
    lineto(Point_E2<T>(B.LL().x(), B.UR().y()));
    closepath();
    fill();
}

template<class T>
void
Postscript<T>::moveto(const Point_E2<T>& p)
{
    PS_Point_E2d pd = world_to_ps_transform(p);
    outfile << pd.x() << " " << pd.y() << " M" << endl;
}

template<class T>
void
Postscript<T>::lineto(const Point_E2<T>& p)
{
    PS_Point_E2d pd = world_to_ps_transform(p);
    outfile << pd.x() << " " << pd.y() << " L" << endl;
}

template<class T>
PS_Point_E2d
Postscript<T>::world_to_ps_transform(const Point_E2<T>& pin)
{
    double worldpsBoxWidth  = to_double(world_bbox.UR().x() - world_bbox.LL().x());
    double worldpsBoxHeight = to_double(world_bbox.UR().y() - world_bbox.LL().y());

    double psboxWidth  = psbox.UR().x() - psbox.LL().x();
    double psboxHeight = psbox.UR().y() - psbox.LL().y();

    double x = psbox.LL().x() + to_double((pin.x() - world_bbox.LL().x()) / worldpsBoxWidth)  * psboxWidth;
    double y = psbox.LL().y() + to_double((pin.y() - world_bbox.LL().y()) / worldpsBoxHeight) * psboxHeight;
    return PS_Point_E2d(x,y);
}

template<typename T>
T
Postscript<T>::world_to_ps_scale(const T& distance)
{
    double worldpsBoxWidth  = to_double(world_bbox.UR().x() - world_bbox.LL().x());
    
    double psboxWidth  = psbox.UR().x() - psbox.LL().x();

    double x = to_double(distance) / worldpsBoxWidth * psboxWidth;

    return x;
}

#endif // POSTSCRIPT_H
