/* The following code example is described in the book "Introduction
 * to Geometric Computing" by Sherif Ghali, Springer-Verlag, 2008.
 *
 * Copyright (C) 2008 Sherif Ghali. This code may be freely copied,
 * modified, or republished electronically or in print provided that
 * this copyright notice appears in all copies. This software is
 * provided "as is" without express or implied warranty; not even for
 * merchantability or fitness for a particular purpose.
 */

#ifndef BSP_E2_SEGMENT_H
#define BSP_E2_SEGMENT_H

#include <ostream>

#include "../../geometry_E2/point_e2.h"
#include "../../geometry_E2/line_e2.h"

// BSP_E2_segment is a model for BSP_geometry::Sub_hyperplane

template<typename NT>
class BSP_E2_segment;

template<typename NT>
std::ostream&
operator<< ( std::ostream& out, const BSP_E2_segment<NT>& s );

template<typename NT>
class BSP_E2_segment
{
private:
    Segment_E2<NT>            _carrying_line;
    BSP_E2_convex_polygon<NT> _bounding_cell;

    mutable bool _source_and_target_are_ready;
    mutable bool _clipped_out;
    mutable Segment_E2<NT> _segment;

public:
    BSP_E2_segment() {}

    BSP_E2_segment( const Segment_E2<NT>& hyperplane,
		    const BSP_E2_convex_polygon<NT>& convex_polytope )
	: _carrying_line( hyperplane ),
	  _bounding_cell( convex_polytope ),
	  _source_and_target_are_ready(false)
    {}

    Segment_E2<NT> carrying_line() const { return _carrying_line; }
    BSP_E2_convex_polygon<NT> bounding_cell() const { return _bounding_cell; }

    void determine_source_and_target() const
    {
	typedef std::pair<bool,Segment_E2<NT> > Bool_and_Segment;
	Bool_and_Segment s = clip_segment_to_cw_convex_region( _carrying_line,
							       _bounding_cell.get_lines() );
	_clipped_out = !s.first;
	if( !_clipped_out )
	    _segment = s.second;
    }

    // precondition for source() and target():
    //   _bounding_cell is finite
    Point_E2<NT> source() const {
	if( ! _source_and_target_are_ready ) {
	    determine_source_and_target();
	    _source_and_target_are_ready = true;
	}
	return _segment.source();
    }

    Point_E2<NT> target() const {
	if( ! _source_and_target_are_ready ) {
	    determine_source_and_target();
	    _source_and_target_are_ready = true;
	}
	return _segment.target();
    }

    friend std::ostream& operator<< <>( ostream& out, const BSP_E2_segment<NT>& s );
};

template<typename NT>
std::ostream&
operator<<( std::ostream& out, const BSP_E2_segment<NT>& s )
{
    out << s._carrying_line;
    return out;
}

#endif // BSP_E2_SEGMENT_H
