/* The following code example is described in the book "Introduction
 * to Geometric Computing" by Sherif Ghali, Springer-Verlag, 2008.
 *
 * Copyright (C) 2008 Sherif Ghali. This code may be freely copied,
 * modified, or republished electronically or in print provided that
 * this copyright notice appears in all copies. This software is
 * provided "as is" without express or implied warranty; not even for
 * merchantability or fitness for a particular purpose.
 */

#ifndef HEDS_H
#define HEDS_H

#include <iostream>
#include <vector>
#include <map>

#include "../geometry_E3/bbox_e3.h"

#include "vertex.h"
#include "halfedge.h"
#include "face.h"

template<typename NumberType, typename FaceType>
class HEDS;
template<typename NumberType, typename FaceType>
struct HEDS_reader;

template<typename NumberType, typename FaceType>
std::ostream& operator<< (std::ostream& os, const HEDS<NumberType, FaceType>& myHEDS);

template<typename NumberType, typename FaceType>
struct HEDS
{
    friend class HEDS_reader<NumberType, FaceType>;

    std::vector< Vertex<NumberType, FaceType>* >   V;
    std::vector< Halfedge<NumberType, FaceType>* > E;
    std::vector< Face<NumberType, FaceType>* >     F;

    Bbox_E3<NumberType> bbox;


    unsigned int numberOfVertices() const { return  V.size(); }
    unsigned int numberOfFaces() const { return F.size(); }

    void stitch_twins();

    void set_vertex_outgoing_edge();
    bool check_twins() const;

    void
    mev(Halfedge<NumberType,FaceType> * adjpred,
	Halfedge<NumberType,FaceType> * adjsucc,
	Vertex<NumberType,FaceType> * vert);

    void
    mef(Halfedge<NumberType,FaceType> * pred,
	Halfedge<NumberType,FaceType> * succ);


    friend
    std::ostream&
    operator<< <>(std::ostream& os, const HEDS<NumberType, FaceType>& myHEDS);
};

template<typename NumberType, typename FaceType>
std::ostream&
operator<< (std::ostream& os, const HEDS<NumberType, FaceType>& myHEDS)
{
    os << "Bounding Box: " << myHEDS.bbox << std::endl << std::endl;

    typedef Vertex<NumberType, FaceType>*                         myVertexP;
    typedef typename std::vector<myVertexP>::const_iterator VertexP_iterator;
    os << "Vertices: " << std::endl;
    for(VertexP_iterator vi = myHEDS.V.begin(); vi!= myHEDS.V.end(); ++vi)
	os << "vertex: " << (*vi)->coords << std::endl;

    // to be completed...

    return os << std::endl;
}

template<typename NumberType, typename FaceType>
void
HEDS<NumberType, FaceType>::stitch_twins()
{
    // ...
    assert(false);
}

template<typename NumberType, typename FaceType>
bool
compare(const Halfedge<NumberType, FaceType>* e1,
	const Halfedge<NumberType, FaceType>* e2)
{
    Vertex<NumberType, FaceType> *e1s,*e1l,*e2s,*e2l;
    if(e1->source < e1->target) {
	e1s = e1->source;	
	e1l = e1->target;
    }
    else{
	e1l = e1->source;	
	e1s = e1->target;
    }
    if(e2->source < e2->target) {
	e2s = e2->source;	
	e2l = e2->target;
    }
    else{
	e2l = e2->source;	
	e2s = e2->target;
    }
    if(e1s == e2s){
	return  e1l < e2l;
    }
    return e1s < e2s;
}

template<typename NumberType, typename FaceType>
bool
isTwin(Halfedge<NumberType, FaceType>* e1,
       Halfedge<NumberType, FaceType>* e2)
{
    if(e1->source == e2->target && e2->source == e1->target)
	return true;
    return false;
}	

template<typename NumberType, typename FaceType>
void
HEDS<NumberType, FaceType>::set_vertex_outgoing_edge()
{
    typedef Vertex<NumberType, FaceType>*                       myVertexP;
    typedef typename std::vector<myVertexP>::iterator     VertexP_iterator;

    typedef Halfedge<NumberType, FaceType>*                   myHalfedgeP;
    typedef typename std::vector<myHalfedgeP>::iterator HalfedgeP_iterator;

    std::map<myVertexP, myHalfedgeP> M;
    for(HalfedgeP_iterator hei = E.begin(); hei!= E.end(); ++hei)
	M[(*hei)->source] = (*hei);

    for(VertexP_iterator vi = V.begin(); vi!= V.end(); ++vi)
	(*vi)->outgoingEdge = M[(*vi)];
}

template<typename NumberType, typename FaceType>
bool
HEDS<NumberType, FaceType>::check_twins() const
{
    typedef typename std::vector<Halfedge<NumberType, FaceType>*>::const_iterator EdgeIt;
    for(EdgeIt it = E.begin(); it != E.end(); ++it)
    {
	if(*it != (*it)->succ->pred)
	    return false;
	if(*it != (*it)->twin->twin)
	    return false;
    }
    return true;
}

template<typename NumberType, typename FaceType>
void
HEDS<NumberType, FaceType>::mev(Halfedge<NumberType,FaceType> * adjpred,
				Halfedge<NumberType,FaceType> * adjsucc,
				Vertex<NumberType,FaceType> * vert)
{
    // makes an edge vertex
    // the vertes is the new vertex
    // the two halfedges are the predecessor and successor of the vertex
    // that will become the new source vertex of the new halfedge
    Halfedge<NumberType,FaceType> * edge =
	new Halfedge<NumberType,FaceType>(adjpred->target,vert,adjpred->adjFace);
    Halfedge<NumberType,FaceType> * twinedge =
	new Halfedge<NumberType,FaceType>(vert,adjsucc->source,adjsucc->adjFace);
    edge->pred = adjpred;
    adjpred->succ = edge;
    edge->succ = twinedge;
    twinedge->pred = edge;
    twinedge->succ = adjsucc;
    adjsucc->pred = twinedge;
    E.push_back(edge);
    E.push_back(twinedge);
}

template<typename NumberType, typename FaceType>
void
HEDS<NumberType, FaceType>::mef(Halfedge<NumberType,FaceType> * pred,
				Halfedge<NumberType,FaceType> * succ)
{
    // this function implements make edge face
    // the pred and succ halfedge pointers are the predecessor
    // and the successor of the half edge that will be created, with
    // new face being adjacent to that half edge 

    Halfedge<NumberType,FaceType> * edge =
	new Halfedge<NumberType,FaceType>(pred->target,succ->source);
    Halfedge<NumberType,FaceType> * twinedge =
	new Halfedge<NumberType,FaceType>(succ->source,pred->target,pred->adjFace);
    twinedge->succ = pred->succ;
    twinedge->pred = succ->pred;
    twinedge->succ->pred = twinedge;
    twinedge->pred->succ = twinedge;

    twinedge->adjFace->e = twinedge;  
    edge->pred = pred;
    edge->succ = succ;
    edge->pred->succ = edge;
    edge->succ->pred = edge;

    Face<NumberType,FaceType> * newFace = new Face<NumberType,FaceType>(edge);
    F.push_back(newFace);
    E.push_back(twinedge);
    E.push_back(edge);
    Halfedge<NumberType,FaceType> * nextedge = edge;
    do {
	nextedge->adjFace = newFace;
	nextedge = nextedge->succ;
    } while(nextedge != edge); 
    Face<NumberType,FaceType> * te = twinedge->adjFace; 
    Point_E3<NumberType> p1,p2,p3,p4;
    newFace->e = edge;
    p1 = newFace->e->source->coords;
    p2 = newFace->e->target->coords;
    p3 = newFace->e->succ->target->coords + Vector_E3<NumberType>(te->normal.dx(), te->normal.dy(), te->normal.dz());
    Direction_E3<NumberType> mydir = Direction_E3<NumberType>(cross_product(Vector_E3<NumberType>(p2,p1),Vector_E3<NumberType>(p2,p3)));
    newFace->normal = mydir;
}

#endif // HEDS_H
