/* The following code example is described in the book "Introduction
 * to Geometric Computing" by Sherif Ghali, Springer-Verlag, 2008.
 *
 * Copyright (C) 2008 Sherif Ghali. This code may be freely copied,
 * modified, or republished electronically or in print provided that
 * this copyright notice appears in all copies. This software is
 * provided "as is" without express or implied warranty; not even for
 * merchantability or fitness for a particular purpose.
 */

#include <cassert>
#include <iostream>
#include <map>

#include "heds.h"
#include "heds_reader.h"

#include "../geometry_E3/point_e3.h"

#include "../misc/color_3.h"

typedef Color_3<float> Color_3f;

typedef float NT;

typedef Point_E3<float> Point_E3f;

typedef HEDS<float, Color_3f>                 HEDS_ff;
typedef HEDS_reader<float, Color_3f>   HEDS_reader_ff;

typedef Vertex<NT, Color_3f>     myVertex;
typedef Halfedge<NT, Color_3f> myHalfedge;
typedef Face<NT, Color_3f>         myFace;

void test_file_reading()
{
    HEDS_ff myHEDS;

    assert( !HEDS_reader_ff::read(myHEDS, "../objects/cube.off") );

    assert( myHEDS.numberOfVertices() == 8 );
    assert( myHEDS.numberOfFaces() == 6 );

    //    assert( myHEDS.check_twins() );
}

void test_stitch_twins_for_a_vertex_pair()
{
    myVertex * V0 = new myVertex(Point_E3f());
    myVertex * V1 = new myVertex(Point_E3f());
	
    myHalfedge *e0 = new myHalfedge(V0, V1);
    myHalfedge *e1 = new myHalfedge(V1, V0);

    HEDS_ff myHEDS;
    myHEDS.E.push_back(e0);
    myHEDS.E.push_back(e1);

    myHEDS.stitch_twins();

    assert( e0->twin == e1 );
    assert( e1->twin == e0 );
}

void test_stitch_twins_for_a_triangular_lamina()
{
    //         V0----------- e3-
    //       /   \ \            |
    //      /    /  \           |
    //  e5 /    /    \          | 
    //    |    e0 F0  -e2------
    //    |   /                  V2
    //     \ /  -----e1--------
    //     V1 /                /
    //        \               /      
    //          --------------
    //                 e4
    // 

    HEDS_ff myHEDS;

    myVertex * V0 = new myVertex(Point_E3f());
    myVertex * V1 = new myVertex(Point_E3f());
    myVertex * V2 = new myVertex(Point_E3f());

    myHEDS.V.push_back(V0); myHEDS.V.push_back(V1); myHEDS.V.push_back(V2);

    myHalfedge *e0 = new myHalfedge(V0, V1);
    myHalfedge *e1 = new myHalfedge(V1, V2);
    myHalfedge *e2 = new myHalfedge(V2, V0);

    myHalfedge *e3 = new myHalfedge(V0, V2);
    myHalfedge *e4 = new myHalfedge(V2, V1);
    myHalfedge *e5 = new myHalfedge(V1, V0);

    e0->succ = e1; e1->succ = e2; e2->succ = e0;
    e3->succ = e4; e4->succ = e5; e5->succ = e3;

    e0->pred = e2; e1->pred = e0; e2->pred = e1;
    e3->pred = e5; e4->pred = e3; e5->pred = e4;

    myHEDS.E.push_back(e0); myHEDS.E.push_back(e1); myHEDS.E.push_back(e2);
    myHEDS.E.push_back(e3); myHEDS.E.push_back(e4); myHEDS.E.push_back(e5);

    myHEDS.stitch_twins();


    assert( e0->twin == e5  && e5->twin == e0 );
    assert( e1->twin == e4  && e4->twin == e1 );
    assert( e2->twin == e3  && e3->twin == e2 );
    
    myHEDS.set_vertex_outgoing_edge();

    assert( V0->outgoingEdge == e0 || V0->outgoingEdge == e3 );
    assert( V1->outgoingEdge == e1 || V1->outgoingEdge == e5 );
    assert( V2->outgoingEdge == e2 || V2->outgoingEdge == e4 );
}

int main()
{
    test_file_reading();
    // test_stitch_twins_for_a_vertex_pair();
    // test_stitch_twins_for_a_triangular_lamina();
}
