/* The following code example is described in the book "Introduction
 * to Geometric Computing" by Sherif Ghali, Springer-Verlag, 2008.
 *
 * Copyright (C) 2008 Sherif Ghali. This code may be freely copied,
 * modified, or republished electronically or in print provided that
 * this copyright notice appears in all copies. This software is
 * provided "as is" without express or implied warranty; not even for
 * merchantability or fitness for a particular purpose.
 */

#ifndef POINT_E2_H
#define POINT_E2_H

#include <iostream>
using std::ostream;
using std::istream;

template<typename T>
class Point_E2;

template<typename T>
ostream& operator<< (ostream& os, const Point_E2<T>& p);

template<typename T>
istream& operator>> (istream& is, Point_E2<T>& p);

template<typename T>
class Point_E2
{
private:
    T _x;
    T _y;
public:
    // Notice that the initialization expressions x() and y() guarantee:
    // 1- that the default constructor for the type T will be used
    // 2- if the type T is a primitive type,
    //    the member variable is initialized to zero.
    Point_E2() : _x(), _y()
    {
    }

    Point_E2(const T& x, const T& y) : _x(x), _y(y) {}

    T x() const { return _x; }
    T y() const { return _y; }

    bool operator==(const Point_E2<T>& p) const {
	return (this == &p) ||
	    (_x == p._x) && (_y == p._y);
    }

    bool operator!=(const Point_E2<T>& p) const {
	return ! operator==(p);
    }

    friend ostream& operator<< <>(ostream& os, const Point_E2<T>& p);

    friend istream& operator>> <>(istream& is, Point_E2<T>& p);
};

template<typename T>
ostream& operator<< (ostream& os, const Point_E2<T>& p)
{
    return os << p._x << " " << p._y;
}

template<typename T>
istream& operator>> (istream& is, Point_E2<T>& p)
{
    return is >> p._x >> p._y;
}

template<typename NT>
NT
squared_distance(const Point_E2<NT>& P, const Point_E2<NT>& Q)
{
    NT x = P.x() - Q.x();
    NT y = P.y() - Q.y();
    return x*x + y*y;
}

#endif // POINT_E2_H
