/* The following code example is described in the book "Introduction
 * to Geometric Computing" by Sherif Ghali, Springer-Verlag, 2008.
 *
 * Copyright (C) 2008 Sherif Ghali. This code may be freely copied,
 * modified, or republished electronically or in print provided that
 * this copyright notice appears in all copies. This software is
 * provided "as is" without express or implied warranty; not even for
 * merchantability or fitness for a particular purpose.
 */

#ifndef VECTOR_E2_H
#define VECTOR_E2_H

#include "point_e2.h"

template<typename T>
class Vector_E2;

template<typename T>
ostream& operator<< (ostream& os, const Vector_E2<T>& v);

template<typename T>
istream& operator>> (istream& is, Vector_E2<T>& v);

template<typename T>
class Vector_E2
{
private:
    T _x;
    T _y;
public:
    Vector_E2() : _x(), _y() {}
    Vector_E2(const T& x, const T& y) : _x(x), _y(y) {}

    T x() const { return _x; }
    T y() const { return _y; }

    bool operator==(const Vector_E2<T>& v) const {
	return (this == &v) ||
	    (_x == v.x()) && (_y == v.y());
    }

    bool operator!=(const Vector_E2<T>& v) const {
	return ! operator==(v);
    }

    friend ostream& operator<< <>(ostream& os, const Vector_E2<T>& p);

    friend istream& operator>> <>(istream& is, Vector_E2<T>& p);
};

template<typename T>
Vector_E2<T>
operator*(const T& d, const Vector_E2<T>& v)
{
    return Vector_E2<T>(d * v.x(), d * v.y());
}

template<typename T>
Vector_E2<T>
operator*(const Vector_E2<T>& v, const T & d)
{
    return Vector_E2<T>(v.x() * d, v.y() * d);
}

template<typename T>
Vector_E2<T>
operator/(const Vector_E2<T>& v, const T & d)
{
    return Vector_E2<T>(v.x() / d, v.y() / d);
}

template<typename T>
Vector_E2<T>
operator+(const Vector_E2<T>& v1, const Vector_E2<T>& v2)
{
    return Vector_E2<T>(v1.x() + v2.x(), v1.y() + v2.y());
}

template<typename T>
Vector_E2<T>
operator-(const Vector_E2<T>& v1, const Vector_E2<T>& v2)
{
    return Vector_E2<T>(v1.x() - v2.x(), v1.y() - v2.y());
}


template<typename T>
Point_E2<T>
operator+(const Point_E2<T>& p, const Vector_E2<T>&  v)
{
    return Point_E2<T>(p.x()+v.x(), p.y()+v.y());
}

template<typename T>
Point_E2<T>
operator-(const Point_E2<T>& p, const Vector_E2<T>&  v)
{
    return Point_E2<T>(p.x()-v.x(), p.y()-v.y());
}

template<typename T>
Vector_E2<T>
operator-(const Point_E2<T>& p1, const Point_E2<T>& p2)
{
    return Vector_E2<T>(p1.x()-p2.x(), p1.y()-p2.y());
}

template<typename T>
ostream& operator<< (ostream& os, const Vector_E2<T>& v)
{
    return os << v._x << " " << v._y;
}

template<typename T>
istream& operator>> (istream& is, Vector_E2<T>& v)
{
    return is >> v._x >> v._y;
}

template<typename T>
T
dot_product(const Vector_E2<T>& v1, const Vector_E2<T>& v2)
{
    return v1.x() * v2.x() + v1.y() * v2.y();
}

template<typename T>
T
cross_product(const Vector_E2<T>& v1, const Vector_E2<T>& v2)
{
    return determinant(v1.x(), v1.y(), v2.x(), v2.y());
}

#endif // VECTOR_E2_H
