/* The following code example is described in the book "Introduction
 * to Geometric Computing" by Sherif Ghali, Springer-Verlag, 2008.
 *
 * Copyright (C) 2008 Sherif Ghali. This code may be freely copied,
 * modified, or republished electronically or in print provided that
 * this copyright notice appears in all copies. This software is
 * provided "as is" without express or implied warranty; not even for
 * merchantability or fitness for a particular purpose.
 */

#ifndef DIRECTION_E3_H
#define DIRECTION_E3_H

#include <cmath>
using std::sqrt;
#include "vector_e3.h"

template<typename T>
class Direction_E3
{
    Vector_E3<T> V;

    void
    normalize() {
	T d = static_cast<T>( sqrt( V.length2() ) );
	V = Vector_E3<T>(V.x()/d, V.y()/d, V.z()/d);
    }

public:
    Direction_E3() : V() {}
    Direction_E3(const T& x, const T& y, const T& z) : V(x,y,z) {
	normalize();
    }
    Direction_E3(const Vector_E3<T>& V) : V(V) {
	normalize();
    }

    T x() const { return V.x(); }
    T y() const { return V.y(); }
    T z() const { return V.z(); }

    Vector_E3<T> getVector_E3() const { return V; }

    Direction_E3 operator-() const { return Direction_E3(-V); }

    Direction_E3<T> operator+(const Direction_E3<T>& D2)
    {
	return Direction_E3<T>(V + D2.V);
    }

    bool operator==(const Direction_E3<T>& d) const {
	return (V == d.V);
    }
};

template<class S>
S
dot_product(const Direction_E3<S>& d1, const Direction_E3<S>& d2)
{
    return dot_product(d1.getVector_E3(), d2.getVector_E3());
}

template<class S>
Direction_E3<S>
cross_product(const Direction_E3<S>& v1, const Direction_E3<S>& v2)
{
    return Direction_E3<S>(+ determinant(v1.y(), v2.y(), v1.z(), v2.z()),
			   - determinant(v1.x(), v2.x(), v1.z(), v2.z()),
			   + determinant(v1.x(), v2.x(), v1.y(), v2.y()));
}

#endif // DIRECTION_E3_H
