/* The following code example is described in the book "Introduction
 * to Geometric Computing" by Sherif Ghali, Springer-Verlag, 2008.
 *
 * Copyright (C) 2008 Sherif Ghali. This code may be freely copied,
 * modified, or republished electronically or in print provided that
 * this copyright notice appears in all copies. This software is
 * provided "as is" without express or implied warranty; not even for
 * merchantability or fitness for a particular purpose.
 */

#ifndef INTERPOLATION_S2_H
#define INTERPOLATION_S2_H

#include "transformation_s2.h"


template<typename NT>
Point_S2<NT>
slerp(const Point_S2<NT>& P1, const Point_S2<NT>& P2, const NT& u)
{
    NT costheta = dot_product(P1.get_Direction_E3(), P2.get_Direction_E3());
    NT theta = std::acos(costheta);
    NT sintheta = std::sin(theta);
    Point_S2<NT> ret = P1.get_Direction_E3() * (std::sin((1-u) * theta) / sintheta)
	+             P2.get_Direction_E3() * (std::sin(u * theta)     / sintheta);
    return ret;
}

template<typename NT>
Rotation_S2<NT>
slerp(const Rotation_S2<NT>& R1, Rotation_S2<NT>& R2, const NT& u)
{
    NT costheta = dot_product(R1.unitq(), R2.unitq());
    NT theta = std::acos(costheta);
    NT sintheta = std::sin(theta);
    Quaternion<NT> retq = R1.unitq() * (std::sin((1-u) * theta) / sintheta)
	+                R2.unitq() * (std::sin(u * theta)     / sintheta);

    return Rotation_S2<NT>(retq);
}

template<typename NT>
Rotation_S2<NT>
lerp(const Rotation_S2<NT>& R1, Rotation_S2<NT>& R2, const NT& t)
{
    Quaternion<NT> Q1 = R1.unitq();
    Quaternion<NT> Q2 = R2.unitq();

    NT unity(static_cast<NT>(1));
    NT r = Q1.r()  * (unity-t) + Q2.r()  * t;
    NT x = Q1.vi() * (unity-t) + Q2.vi() * t;
    NT y = Q1.vj() * (unity-t) + Q2.vj() * t;
    NT z = Q1.vj() * (unity-t) + Q2.vk() * t;
    return Rotation_S2<NT>(r, Direction_E3<NT>(x, y, z));
}

#endif // INTERPOLATION_S2_H
