/* The following code example is described in the book "Introduction
 * to Geometric Computing" by Sherif Ghali, Springer-Verlag, 2008.
 *
 * Copyright (C) 2008 Sherif Ghali. This code may be freely copied,
 * modified, or republished electronically or in print provided that
 * this copyright notice appears in all copies. This software is
 * provided "as is" without express or implied warranty; not even for
 * merchantability or fitness for a particular purpose.
 */

#ifndef GLDRAW_E2_H
#define GLDRAW_E2_H

#include <vector>

#include "glow.h"
#include "glowAux.h"
using namespace glow;

#include "../misc/conversions.h"
#include "../geometry_E2/point_e2.h"

template<typename T>
struct GLdraw_E2
{
    typedef Point_E2<int>        Point_E2i;
    typedef Point_E2<double>     Point_E2d;

    static void draw_point(const Point_E2<T>& p,
			   const GlowColor& pointColor = GlowColor::yellow,
			   float pointSize = 7.0)
    {
	glPointSize(pointSize);
	pointColor.Apply();
	glBegin(GL_POINTS); {
	    glVertex2f( to_float(p.x()), to_float(p.y()) );
	} glEnd();
    }

    static void draw_segment(const Point_E2<T>& source, const Point_E2<T>& target,
			     const GlowColor& segmentColor = GlowColor::cyan,
			     float lineWidth = 3.0,
			     bool drawEndpoints = true,
			     const GlowColor& pointColor = GlowColor::yellow,
			     float pointSize = 7.0)
    {
	glLineWidth(lineWidth);

	segmentColor.Apply();
	glBegin(GL_LINES); {
	    glVertex2f( to_float(source.x()), to_float(source.y()) );
	    glVertex2f( to_float(target.x()), to_float(target.y()) );
	} glEnd();

	if(drawEndpoints) {
	    draw_point(source, pointColor, pointSize);
	    draw_point(target, pointColor, pointSize);
	}
    }

    static void draw_polygon(const std::vector<Point_E2<T> >& vertices,
			     const GlowColor& segmentColor = GlowColor::cyan,
			     const GlowColor& pointColor = GlowColor::yellow,
			     float lineWidth = 3.0,
			     float pointSize = 7.0)
    {
	for(unsigned int i = 0; i != vertices.size(); ++i)
	    draw_segment( vertices[i],
			  vertices[(i+1)%vertices.size()],
			  segmentColor,
			  lineWidth,
			  true,
			  pointColor,
			  pointSize);
    }

    static void draw_rubber_polygon(const std::vector<Point_E2<T> >& vertices,
				    const Point_E2<T>& rubberPoint)
    {
	for(unsigned int i = 0; i != vertices.size()-1; ++i)
	    draw_segment( vertices[i], vertices[i+1] );

	draw_segment( vertices[0], rubberPoint );
	draw_segment( rubberPoint, vertices[vertices.size()-1] );
    }

    static void clear_screen()
    {
	glClear(GL_COLOR_BUFFER_BIT);
    }

    static void set_color(float r, float g, float b)
    {
	glColor3f(r,g,b);
    }

    static void setup_projection(const Point_E2d& wmin, const Point_E2d& wmax)
    {
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	gluOrtho2D(wmin.x(), wmax.x(), wmin.y(), wmax.y());

	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();

	glDisable(GL_DEPTH_TEST);
	glDisable(GL_LIGHTING);

	glEnable(GL_LINE_SMOOTH);
	glEnable(GL_POINT_SMOOTH);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA); 

	glColor3f(0.0f, 0.0f, 0.0f);
	glLineWidth(3.0);
	glPointSize(7.0);
    }

    static Point_E2d unproject(const Point_E2i& pin)
    {
	GLint viewport[4];
	GLdouble mvmatrix[16], projmatrix[16];
	GLint realy;		//  OpenGL y coordinate position
	GLdouble wx, wy, wz;	// returned world x, y, z coords

	glGetIntegerv(GL_VIEWPORT, viewport);
	glGetDoublev(GL_MODELVIEW_MATRIX, mvmatrix);
	glGetDoublev(GL_PROJECTION_MATRIX, projmatrix);

	// viewport[3] is the height of the window in pixels.
	realy = viewport[3] - (GLint) pin.y() - 1;

	gluUnProject((GLdouble) pin.x(), (GLdouble) realy,
		     0.0, 	// Ortho2D means any z-value will do
		     mvmatrix, projmatrix, viewport, &wx, &wy, &wz); 

	return Point_E2d(wx, wy);
    }
};

#endif // GLDRAW_E2_H
