/* The following code example is described in the book "Introduction
 * to Geometric Computing" by Sherif Ghali, Springer-Verlag, 2008.
 *
 * Copyright (C) 2008 Sherif Ghali. This code may be freely copied,
 * modified, or republished electronically or in print provided that
 * this copyright notice appears in all copies. This software is
 * provided "as is" without express or implied warranty; not even for
 * merchantability or fitness for a particular purpose.
 */

#include "point_e2.h"
#include "bbox_e2.h"
#include "point_e3.h"
#include "projection_e3.h"

#include "normalize_e3.h"
#include "transformation_e3.h"

#include "point_s2.h"
#include "circle_s2.h"

#include "postscript.h"

template<typename NT>
class PSdraw_S2
{
    Postscript<NT> myPS;
    Transformation_E3<NT> T;
    Project_on_xy<NT> myProjection;

    Point_S2<NT> viewer_S2;
public:
    PSdraw_S2(const char* psfilename,
	      const Point_E3<NT>& viewer,
	      const Point_E3<NT>& coi,
	      const Bbox_E2<NT>& bbox)
    	: myPS(psfilename, bbox),
	  viewer_S2(viewer.x(), viewer.y(), viewer.z())
    {
	T = Transformation_E3<NT>(ORTHOGONAL, viewer, coi);
	myProjection = Project_on_xy<NT>();

	myPS.set_line_width(3.0);

	myPS.set_line_cap(0);
	myPS.set_line_join(1);
    }

    virtual ~PSdraw_S2() {
	draw(viewer_S2, false);
    }

    void draw( const Circle_S2<NT>& circle,
	       bool distinct_back_faces = true,
	       double back_intensity = 0.7 )
    {
	double black = 0.0;
	double gray = distinct_back_faces ? back_intensity : black;

	Point_S2<NT> base1 = circle.base1();
	Point_S2<NT> base2 = circle.base2();

	NT lip = inner_product( base1, viewer_S2 );
	bool last_front_facing = (lip > 0.0);

	myPS.set_gray( last_front_facing ? black : gray );
	myPS.moveto( my_projection(base1, T, myProjection) );

	for(double theta = 0.0; theta <= 2*M_PI + M_PI/60.0; theta += M_PI/60.0)
	{
	    Point_S2<NT> ps2 = base1 * NT(cos(theta)) + base2 * NT(sin(theta));

	    NT ip = inner_product( ps2, viewer_S2 );

	    bool front_facing = (ip > 0.0);
	    Point_E2<NT> pe2 = my_projection(ps2, T, myProjection);

	    if(front_facing != last_front_facing) {
		myPS.lineto( pe2 );
		myPS.stroke();
		myPS.set_gray( front_facing ? black : gray );
		myPS.moveto( pe2 );
	    }
	    else
		myPS.lineto( pe2 );

	    last_front_facing = front_facing;
	}
	myPS.stroke();
    }

private:
    Point_E2<NT>
    my_projection(const Point_S2<NT>& p,
		  const Transformation_E3<NT>& T,
		  const Project_on_xy<NT>& myProjection)
    {
	Vector_E3<NT> ve3(p.x(), p.y(), p.z());
	Vector_E3<NT> ve3nf;
	normalize(ve3, ve3nf);

	Point_E3<NT> pe3 = Point_E3<NT>(0,0,0) + ve3nf;
	Point_E3<NT> pt = T(pe3);
	Point_E2<NT> p2d = myProjection( pt );

	return p2d;
    }
};
