/* The following code example is described in the book "Introduction
 * to Geometric Computing" by Sherif Ghali, Springer-Verlag, 2008.
 *
 * Copyright (C) 2008 Sherif Ghali. This code may be freely copied,
 * modified, or republished electronically or in print provided that
 * this copyright notice appears in all copies. This software is
 * provided "as is" without express or implied warranty; not even for
 * merchantability or fitness for a particular purpose.
 */

#ifndef BSP_E2_H
#define BSP_E2_H

#include "bsp_e2_convex_polygon.h"
#include "../../geometry_E2/predicates_e2.h"
#include "../../geometry_E2/intersection_e2.h"

template<typename BSP_geometry,
	 typename Boundary_attributes,
	 typename Interior_attributes>
class BSP_node;

//----------------------------------------------------------------
template<
    typename NT,
    typename Boundary_attributes,
    typename Interior_attributes>
class BSP_E2
{
public:

    typedef Segment_E2<NT>  Hyperplane;
    typedef Point_E2<NT>  Point;
    typedef BSP_E2_convex_polygon<NT>  BSP_convex_polytope;
    typedef BSP_E2_segment<NT>  Sub_hyperplane;

    typedef BSP_E2<NT, Boundary_attributes, Interior_attributes> BSP_geometry;
    typedef BSP_node<BSP_geometry, Boundary_attributes, Interior_attributes>   BSP_node_E2;
//----------------------------------------------------------------
public:

    // A triangle is represented using the following subtree.             !  
    //                                                                    !  
    //                    o (splittingLine = LP[0])          node 1       !  
    //                 - / \ +                                            !  
    //                  /   \                                             !  
    //                 o     F (splittingLine = LP[1])       nodes 2, 3   !  
    //              - / \ +                                               !  
    //               /   \                                                !  
    //              o     F (splittingLine = LP[2])          nodes 4, 5   !  
    //           - / \ +                                                  !  
    //            /   \                                                   !  
    //           T     F                                     nodes 6, 7   !  
    //                                                                    !  
    //  o : interior node                                                 !  

    //----------------------------------------------------------------
    static void
    build_subtree(
		  BSP_node_E2 * current_node,
		  const BSP_E2_convex_polygon<NT>& P,
		  const Boundary_attributes& _boundary_attributes,
		  const Interior_attributes& _interior_attributes,
		  bool reverseSpace = false )
    {
	std::vector<Segment_E2<NT> > bounding_lines = P.get_free_bounding_lines();

	if( bounding_lines.size() > 0 ) {
	    typedef typename std::vector<Segment_E2<NT> >::iterator ExtSegIt;
	    ExtSegIt lineIt = bounding_lines.begin();
	    current_node->set_interior_node( *lineIt, _boundary_attributes );
	    BSP_node_E2* current = current_node;
	    if( bounding_lines.size() == 1 ) {
		current->positive_child = new BSP_node_E2( reverseSpace );
		if( reverseSpace )
		    current->positive_child->interior_attributes = _interior_attributes;
		current->negative_child = new BSP_node_E2( !reverseSpace );
		if( !reverseSpace )
		    current->negative_child->interior_attributes = _interior_attributes;
	    }
	    else {
		do {
		    ++lineIt;
		    current->positive_child = new BSP_node_E2( reverseSpace );
		    if( reverseSpace )
			current->positive_child->interior_attributes = _interior_attributes;
		    current->negative_child = new BSP_node_E2(false);
		    if(lineIt != bounding_lines.end())
			current->negative_child->set_interior_node( *lineIt, _boundary_attributes );
		    else
			current->negative_child->set_leaf_node( !reverseSpace, _interior_attributes );
		    current = current->negative_child;
		} while( lineIt != bounding_lines.end() );
	    }
	}
    }
    //----------------------------------------------------------------


    static void
    split( const BSP_E2_convex_polygon<NT>& a_polygon,
	   const Segment_E2<NT> splitting_line,
	   BSP_E2_convex_polygon<NT>& positive_side,
	   BSP_E2_convex_polygon<NT>& negative_side )

	// We use a Segment_2 object for the splitting line because
	// Line_E2 does not maintain a source/target pair (it may have
	// been defined directly using coefficients.)

    {
	const std::vector<Segment_E2<NT> > a_polygon_lines = a_polygon.get_lines();
	const std::vector<bool> a_polygon_line_is_closed = a_polygon.get_line_is_closed();

	{
	    // quick exit
	    if (a_polygon.is_empty()) {
		positive_side = BSP_E2_convex_polygon<NT>(); // empty set
		negative_side = BSP_E2_convex_polygon<NT>(); // empty set
		return;
	    }
	}

	std::vector<Point_E2<NT> > points;

	{
	    // Determine polygon boundary points

	    const std::vector<Segment_E2<NT> >& lines = a_polygon_lines;

	    Point_E2<NT>  ipoint = intersection_of_lines(*(lines.end() - 1), *(lines.begin()));
	    points.push_back(ipoint);

	    typedef typename std::vector<Segment_E2<NT> >::const_iterator ESit;

	    for(ESit I = lines.begin();
		I != lines.end()-1; ++I)
	    {
		Point_E2<NT>  ipoint = intersection_of_lines(*I, *(I+1));
		points.push_back(ipoint);
	    }
	}

	bool all_on_positive_side = true;
	bool all_on_oriented_boundary = true;
	bool all_on_negative_side = true;

	typedef typename std::vector<Point_E2<NT> >::const_iterator EPit;
	for(EPit vi = points.begin();
	    vi != points.end(); ++vi)
	{

	    int s = oriented_side( splitting_line, *vi );

	    switch(s) {
	    case ON_POSITIVE_SIDE:
		all_on_negative_side = false;
		all_on_oriented_boundary = false;
		break;
	    case ON_ORIENTED_BOUNDARY:
		break;
	    case ON_NEGATIVE_SIDE:
		all_on_positive_side = false;
		all_on_oriented_boundary = false;
		break;
	    }
	}

	if(all_on_positive_side || all_on_negative_side) {

	    // To ensure there are no duplicates in any path from root to leaf,
	    // we reset the bits corresponding to edges that coincide with the partitioning line.

	    // If we wish to handle concave polygons as well, the next block should move
	    // outside the enclosing if-statement. This makes it possible to handle a
	    // concave polygon with one or more edges lying on the splitting line.

	    std::vector<bool> edgeFlags = a_polygon.get_line_is_closed();

	    {

		std::vector<bool>::iterator edgeFlagIt = edgeFlags.begin();

		typedef typename std::vector<Segment_E2<NT> >::const_iterator ESit;
		for(ESit lit = a_polygon_lines.begin();
		    lit != a_polygon_lines.end(); ++lit)
		{
		    // Does this carrying line coincide with the partitioning line?
		    // ----------------
		    // We answer line equality by checking for segment endpoint coincidence.

		    if( oriented_side(*lit, splitting_line.source()) == ON_ORIENTED_BOUNDARY &&
			oriented_side(*lit, splitting_line.target()) == ON_ORIENTED_BOUNDARY )
			*edgeFlagIt = false; // true;

		    ++edgeFlagIt;
		}
	    }

	    if(all_on_positive_side) // case (1)
	    {
		positive_side = BSP_E2_convex_polygon<NT>( a_polygon_lines, edgeFlags );
		/*, a_polygon.attributes() */
		negative_side = BSP_E2_convex_polygon<NT>(); // the empty set

		return;
	    }

	    if(all_on_negative_side) // case (2)
	    {
		positive_side = BSP_E2_convex_polygon<NT>(); // the empty set
		negative_side = BSP_E2_convex_polygon<NT>( a_polygon_lines, edgeFlags );
		/*,a_polygon.attributes() */

		return;
	    }
	}

	if(all_on_oriented_boundary) // case (3)
	{
	    positive_side = BSP_E2_convex_polygon<NT>(); // the empty set
	    negative_side = BSP_E2_convex_polygon<NT>(); // the empty set

	    return;
	}

	// assert(static_cast<int>(sides.size()) == a_polygon.polygon.size());

	std::vector<Segment_E2<NT> > positiveLines;  std::vector<bool> positiveFlags;
	std::vector<Segment_E2<NT> > negativeLines;  std::vector<bool> negativeFlags;

	typedef typename std::vector<bool>::const_iterator Bit;
	Bit edgeFlagI = a_polygon_line_is_closed.begin();

	typename std::vector<Point_E2<NT> >::const_iterator pi = points.begin();
	typename std::vector<Point_E2<NT> >::const_iterator pj = pi + 1;

	typedef typename std::vector<Segment_E2<NT> >::const_iterator ESit;
	for(ESit eit = a_polygon_lines.begin();
	    eit != a_polygon_lines.end();
	    ++eit)
	{
	    int vertexI = oriented_side(splitting_line, *pi);
	    int vertexJ = oriented_side(splitting_line, *pj);

	    //----------------------------------------------------------------
	    // Segment is on one side of the plane only (possibly touching).
	    // The following correctly handles the case when both points are ON_ORIENTED_BOUNDARY
	    //----------------------------------------------------------------
	    if( vertexI != ON_NEGATIVE_SIDE &&
		vertexJ != ON_NEGATIVE_SIDE )
	    {
		// case (4)
		positiveLines.push_back( *eit );
		positiveFlags.push_back( *edgeFlagI );

		if( vertexJ == ON_ORIENTED_BOUNDARY ) {
		    positiveLines.push_back( splitting_line );
		    positiveFlags.push_back( true );
		}
	    }
	    else if( vertexI != ON_POSITIVE_SIDE &&
		     vertexJ != ON_POSITIVE_SIDE )
	    {
		// case (6)
		negativeLines.push_back( *eit );
		negativeFlags.push_back( *edgeFlagI );

		if( vertexJ == ON_ORIENTED_BOUNDARY ) {
		    negativeLines.push_back( Segment_E2<NT> (splitting_line.target(),
							      splitting_line.source()) );
		    negativeFlags.push_back( true );
		}
	    }
	    else
	    {
		//----------------------------------------------------------------
		// Segment straddles the plane; split
		//----------------------------------------------------------------

		// // The following assertion must hold if the input polygons are convex
		// assert((*vertexI == ON_POSITIVE_SIDE && *vertexJ == ON_NEGATIVE_SIDE) ||
		//        (*vertexI == ON_NEGATIVE_SIDE && *vertexJ == ON_POSITIVE_SIDE));

		Point_E2<NT>  ipoint = intersection_of_lines( *eit, splitting_line );

		// splitting line appears on both +ve and -ve sides

		// but we treat adjacency differently depending whether we are going

		// from +ve to -ve
		if(vertexI == ON_POSITIVE_SIDE && vertexJ == ON_NEGATIVE_SIDE) // case (8)
		{
		    positiveLines.push_back( *eit );
		    positiveFlags.push_back( *edgeFlagI );

		    // reverse orientation (polygons are oriented *clockwise*):
		    positiveLines.push_back( Segment_E2<NT> (splitting_line.target(),
							     splitting_line.source()) );
		    // edges resulting from splits have partitioning_line_encountered==true
		    positiveFlags.push_back( false ); // true );

		    negativeLines.push_back( *eit );
		    negativeFlags.push_back( *edgeFlagI );
		}

		// or from -ve to +ve
		if(vertexI == ON_NEGATIVE_SIDE && vertexJ == ON_POSITIVE_SIDE) // case (9)
		{
		    negativeLines.push_back( *eit );
		    negativeFlags.push_back( *edgeFlagI );

		    negativeLines.push_back( splitting_line );
		    negativeFlags.push_back( false ); // true );

		    positiveLines.push_back( *eit );
		    positiveFlags.push_back( *edgeFlagI );
		}
	    }

	    ++edgeFlagI;
	    ++pi;

	    if(pi + 1 == points.end())
		pj = points.begin();
	    else
		pj = pi + 1;
	}

	positive_side = BSP_E2_convex_polygon<NT>( positiveLines, positiveFlags /*, a_polygon.attributes()*/ );
	negative_side = BSP_E2_convex_polygon<NT>( negativeLines, negativeFlags /*, a_polygon.attributes()*/ );
    }

    //----------------------------------------------------------------

    static void 
    construct_sub_hyperplane( const Hyperplane& hyperplane,
			      const BSP_convex_polytope& convex_polytope,
			      Sub_hyperplane& sub_hyperplane)
    {
	sub_hyperplane = BSP_E2_segment<NT>( hyperplane, convex_polytope );
    }
};

#endif // BSP_E2_H
