/* The following code example is described in the book "Introduction
 * to Geometric Computing" by Sherif Ghali, Springer-Verlag, 2008.
 *
 * Copyright (C) 2008 Sherif Ghali. This code may be freely copied,
 * modified, or republished electronically or in print provided that
 * this copyright notice appears in all copies. This software is
 * provided "as is" without express or implied warranty; not even for
 * merchantability or fitness for a particular purpose.
 */

#ifndef BSP_S1_H
#define BSP_S1_H

#include "bsp_s1_segment.h"

template<typename BSP_geometry,
	 typename Boundary_attributes,
	 typename Interior_attributes>
class BSP_node;

template<typename NT,
	 typename Boundary_attributes,
	 typename Interior_attributes>
class BSP_S1
{
public:

    typedef Point_S1<NT>         Hyperplane;
    typedef Point_S1<NT>         Sub_hyperplane;
    typedef Point_S1<NT>         Point;
    typedef BSP_S1_segment<NT>   BSP_convex_polytope;

    typedef BSP_S1<NT, Boundary_attributes, Interior_attributes> BSP_geometry;
    typedef BSP_node<BSP_geometry, Boundary_attributes, Interior_attributes>   BSP_node_S1;

public:
    static void build_subtree( BSP_node_S1 * current_node,
			       const BSP_S1_segment<NT>& I,
			       const Boundary_attributes& _boundary_attributes,
			       const Interior_attributes& _interior_attributes,
			       bool reverse_space = false )
    // assume operation is union and reverse_space is false on first reading
    {
	if( I.is_source_closed() )
	    if( I.is_target_closed() ) {

		if( I.source() != I.target().antipode() ) {
		    current_node->set_interior_node( I.target(), _boundary_attributes );

		    current_node->negative_child = new BSP_node_S1( I.source(), _boundary_attributes );

		    current_node->positive_child = new BSP_node_S1( reverse_space, _interior_attributes );
		    current_node->negative_child->negative_child = new BSP_node_S1( reverse_space, _interior_attributes );
		    current_node->negative_child->positive_child = new BSP_node_S1( !reverse_space, _interior_attributes );
		}
		else {
		    current_node->set_interior_node( I.target(), _boundary_attributes );
		    current_node->positive_child = new BSP_node_S1(  reverse_space, _interior_attributes );
		    current_node->negative_child = new BSP_node_S1( !reverse_space, _interior_attributes );
		}
	    }
	    else {
		current_node->set_interior_node( I.source(), _boundary_attributes );
		current_node->negative_child = new BSP_node_S1(  reverse_space, _interior_attributes );
		current_node->positive_child = new BSP_node_S1( !reverse_space, _interior_attributes );
	    }
	else
	    if( I.is_target_closed() ) {
		current_node->set_interior_node( I.target(), _boundary_attributes );
		current_node->positive_child = new BSP_node_S1(  reverse_space, _interior_attributes );
		current_node->negative_child = new BSP_node_S1( !reverse_space, _interior_attributes );
	    }
	    else
		current_node->set_leaf_node( !reverse_space, _interior_attributes );
    }

    static void split(const BSP_S1_segment<NT>& I,
		      const Point_S1<NT> hyperplane,
		      BSP_S1_segment<NT>& positive_part,
		      BSP_S1_segment<NT>& negative_part)
    {
	if( I.is_full() ) {
	    positive_part = BSP_S1_segment<NT>(hyperplane, hyperplane.antipode());
	    negative_part = BSP_S1_segment<NT>(hyperplane.antipode(), hyperplane);
	    return;
	}

	const Oriented_side source_of_segment = oriented_side( hyperplane, I.source() );
	const Oriented_side target_of_segment = oriented_side( hyperplane, I.target() );

	// Cases c
	if( source_of_segment != ON_NEGATIVE_SIDE &&
	    target_of_segment != ON_NEGATIVE_SIDE )
	{
	    negative_part.set_empty( true );
	    positive_part = I;
	    // Case c1
	    if( source_of_segment == ON_ORIENTED_BOUNDARY )
		positive_part.set_source_closed( false );
	    // Case c3
	    if( target_of_segment == ON_ORIENTED_BOUNDARY )
		positive_part.set_target_closed( false );
	    return;
	}

	// Cases a
	if( source_of_segment != ON_POSITIVE_SIDE &&
	    target_of_segment != ON_POSITIVE_SIDE )
	{
	    positive_part.set_empty( true );
	    negative_part = I;
	    // case a1
	    if( source_of_segment == ON_ORIENTED_BOUNDARY )
		negative_part.set_source_closed( false );
	    // case a3
	    if( target_of_segment == ON_ORIENTED_BOUNDARY )
		negative_part.set_target_closed( false );
	    return;
	}

	// Case b: split by point
	if( source_of_segment == ON_NEGATIVE_SIDE
	    && target_of_segment == ON_POSITIVE_SIDE )
	{
	    negative_part = BSP_S1_segment<NT>( I.source(), hyperplane, true, false ); // closed, open
	    positive_part = BSP_S1_segment<NT>( hyperplane, I.target(), false, true ); // open, closed
	}
	// Case d: split by antipode of point
	if( source_of_segment == ON_POSITIVE_SIDE
	    && target_of_segment == ON_NEGATIVE_SIDE )
	{
	    positive_part = BSP_S1_segment<NT>( I.source(), hyperplane.antipode(),
						true, false ); // closed, open
	    negative_part = BSP_S1_segment<NT>( hyperplane.antipode(), I.target(),
						false, true ); // open, closed
	}
    }
};

#endif // BSP_S1_H
