/* The following code example is described in the book "Introduction
 * to Geometric Computing" by Sherif Ghali, Springer-Verlag, 2008.
 *
 * Copyright (C) 2008 Sherif Ghali. This code may be freely copied,
 * modified, or republished electronically or in print provided that
 * this copyright notice appears in all copies. This software is
 * provided "as is" without express or implied warranty; not even for
 * merchantability or fitness for a particular purpose.
 */

#include <fstream>
#include <string>
using std::string;

template<typename NumberType, typename FaceType>
struct HEDS_reader
{
    // return false means file was read successfully
    static bool read(HEDS<NumberType, FaceType>& heds,
		     const string& filename,
		     bool invert = false // by default polygons are not reversed
		     )
    {
	std::ifstream in( filename.c_str() );

	if ( !in) {
	    std::cerr << "Cannot open '"<< filename << std::endl;
	    return true;
	}

	string offident;
	in >> offident;
	if(offident != "OFF") {
	    std::cerr << "OFF file does not start with keyword 'OFF'\n";
	    return true;
	}

	int numV, numF, unusedNumE;
	in >> numV >> numF >> unusedNumE;

	double x,y,z;
	in >> x >> y >> z;

	Point_E3<NumberType> myP(static_cast<NumberType>(x),
				 static_cast<NumberType>(y),
				 static_cast<NumberType>(z));

	heds.V.push_back( new Vertex<NumberType, FaceType>(myP) );
	heds.bbox = Bbox_E3<NumberType>(myP);

	for(int i=1; i<numV; i++) {
	    in >> x >> y >> z;
	    Point_E3<NumberType> myP2(static_cast<NumberType>(x),
				      static_cast<NumberType>(y),
				      static_cast<NumberType>(z));

	    heds.V.push_back( new Vertex<NumberType, FaceType>(myP2) );
	    heds.bbox += myP2;
	}
	for(int f=0; f<numF; f++) {
	    int numSides;
	    in >> numSides;

	    int v0, v1;
	    in >> v0 >> v1;
	    
	    // Create e01, but we can't set its pred or its pred's succ

	    // set source and target
	    Halfedge<NumberType, FaceType> *e01 =
		new Halfedge<NumberType, FaceType>(heds.V[v0], heds.V[v1]);
	    heds.E.push_back(e01);
	    Face<NumberType, FaceType> *face = new Face<NumberType, FaceType>(e01);
	    // face->set_color(Color(0.5,0.5,0.5));
	    heds.F.push_back(face);
	    e01->adjFace = face;

	    int lastV = v1;
	    Halfedge<NumberType, FaceType> *lastE = e01;
	    for(int v=2; v<numSides; v++) {
		int vindex;
		in >> vindex;
		// Create e and set its pred (e->pred) and lastE->succ
		Halfedge<NumberType, FaceType> *e =
		    new Halfedge<NumberType, FaceType>(heds.V[lastV], heds.V[vindex], face);
		heds.E.push_back(e);
		e->pred = lastE;
		lastE->succ = e;

		lastV = vindex;
		lastE = e;
	    }

	    // Create e_numSides_0 and set succ/pred
	    Halfedge<NumberType, FaceType> *eclose =
		new Halfedge<NumberType, FaceType>(heds.V[lastV], heds.V[v0], face);
	    heds.E.push_back(eclose);
	    eclose->pred = lastE;
	    lastE->succ = eclose;

	    // Now set e01's pred
	    e01->pred = eclose;
	    eclose->succ = e01;
	}    
	
	// heds.stitch_twins();

	heds.set_vertex_outgoing_edge();

	if ( !in) { 
	    std::cerr << "offview: read error: while reading file '" << filename << "'." << std::endl;
	    return true;
	}
	return false;
    }
};
