## estimate volatility for all currencies with three adaptive methods
## and GARCH
library( tseries)
library( zoo)
library( lcplmsssa)
library( lattice)

exch.rates <- read.zoo("exch-rates.csv", format = "%F", header = TRUE)

log.returns <- diff( log( na.locf( exch.rates[,"GBP"])))

## make it up to 250 or 500 (1 or 2 years)
interval.length <- c( 5,7,10,13,16,20,24,30)

horizons <- c(1, 5, 10)

#########################################################
compute.msqe <- local({
  variance.realized <-
    do.call( rbind, lapply( horizons, function( horizon){
      bzz <- rollapply( log.returns^2
                       , width = horizon
                       , FUN = sum
                       , align = "left")
      data.frame( horizon = horizon, date = index( bzz),
                 realized.vola = coredata( bzz))
    }))

  ## return a function computing a MSqE
  function( forecast, h){
    ## compute squared error
    ## implicitly merging two zoo objects
    sqe <- sqrt( abs( forecast - local({
      zz <- subset( variance.realized, horizon == h,
                   select = c( date, realized.vola))
      zoo( zz$realized.vola, order.by = zz$date)
    })))
    ## aggregate by years
    year <- cut( index( sqe), breaks = "year", labels = NULL)
    levels( year) <- format( as.Date(levels( year)), "%Y")
    aggregate( sqe, year, mean) ## FIXME: was sum instead of mean
  }
})

## adaptive methods (all of them)
criterion.ada <- do.call( rbind, lapply( c("LCP","LMS", "SSA"), function(m){
  bzz <- filter.lcplmsssa( method = m
                          , model = "Volatility"
                          , interval.length = interval.length
                          , aggregation.kernel = "Quadratic"
                          , rpower = 0.5
                          , test.level = 0.2
                          , data = log.returns^2 )$estimate
  do.call( rbind, lapply( horizons, function( h){
    msqe <- compute.msqe( zoo( h * bzz, order.by = tail( index( log.returns),
                                          length( bzz))), h)
    data.frame( method = m
               , horizon = h
               , date = index( msqe)
               , criterion = coredata( msqe))
  }))
}))

## GARCH
criterion.garch <- local({
  variance.garch <-
    rollapply(
              ## approximate missing values
              as.zooreg( na.approx( log.returns)),
              width = tail( interval.length, 1),
              align = "right",
              function( x){
                gg <- garch( na.remove( as.ts( x)),
                            by.column = TRUE, trace = FALSE)
                
                cc <- coef( gg)
                uncond.var <- cc["a0"] / ( 1 - cc["a1"] - cc["b1"])
                cond.var <- tail( fitted( gg)[,"sigt"], 1)^2
                
                ans <- 
                  cumsum( sapply( 1:max( horizons), function( h){
                    uncond.var + ( cc["a1"] + cc["b1"])^h * (cond.var -
                                                             uncond.var)
                  }))[ horizons]
                names( ans) <-
                  paste("forecast", as.character( horizons), sep = ".")
                ans
              })

  do.call( rbind, lapply( horizons, function( h){
    msqe <- compute.msqe( variance.garch[, paste("forecast", h, sep=".")], h)
    data.frame( method = "GARCH"
               , horizon = h
               , date = index( msqe)
               , criterion = coredata( msqe))
  }))
})

zz <- rbind( criterion.ada, criterion.garch)

trellis.device( device = "pdf", file = "XFGadamethperf.pdf",
               color = FALSE, ## why are books not colourful?
               height = 4, width = 8 )
## get rid of the grey background in headers
mypar <- trellis.par.get("strip.background")
mypar$col <- "white"
trellis.par.set( "strip.background" = mypar)
print( dotplot( criterion ~ method | date,
        data = zz, horizontal = FALSE,
        abbreviate = TRUE, auto.key = list( title = "Horizon"),
        groups = horizon,
        layout = c( nlevels( zz$date), 1),
        ylab = "MSqE",
        ## rotate x-labels, for long they are
        scales = list( rot = c(90, 0))))
dev.off()
