

#include <linux/version.h>	/* Linux Version */
#include <linux/module.h>	/* Makros and Defines */
#include <linux/fs.h>
#include <asm/uaccess.h>	/* put_user() */
#include <linux/cdev.h>
#include <linux/sched.h>	/* struct task_struct *current */
#include <linux/ioport.h>
#include <asm/io.h>

#define MAJORNUM 60
#define NUMDEVICES 1
#define DEVNAME "blink"
#define PORT 0x378

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Joachim Schroeder");
MODULE_DESCRIPTION("File Operations Example");

/* Prototypes */
int fops_init(void);
void fops_exit(void);
static ssize_t fops_read(struct file *, char *, size_t, loff_t *);
static ssize_t fops_write(struct file *, const char *, size_t, loff_t *);
static int fops_open(struct inode *, struct file *);
static int fops_close(struct inode *, struct file *);
static void timer_func(unsigned long ptr);

/* Global Variables */
static int is_open = 0;
static int new_read = 0;
static char freq = 1;
static char val;
static int minor_num;

static struct cdev *driver_info = NULL;
static struct file_operations fops = {
	.owner		= 	THIS_MODULE,
	.read		=	fops_read,
	.write		=	fops_write,
	.open		=	fops_open,
	.release	=	fops_close
};
static struct timer_list timer;

int fops_init(void) {
	printk(KERN_INFO "Module blink: init()\n");

	// allocate device number
	if( register_chrdev_region( MKDEV(MAJORNUM,0), NUMDEVICES, DEVNAME ) ) {
		pr_debug("Device number 0x%02x not available ...\n", MKDEV(MAJORNUM,0));
		return -EIO;
	}

	// allocate device
	driver_info = cdev_alloc();
	if( driver_info == NULL ) {
		pr_debug("cdev_alloc failed!\n");
		goto free_devnum;
	}

	// specify device structure (init) and register (add)
	kobject_set_name(&driver_info->kobj, DEVNAME );
	driver_info->owner = THIS_MODULE;
	cdev_init( driver_info, &fops );
	if( cdev_add( driver_info, MKDEV(MAJORNUM,0), NUMDEVICES) ) {
		pr_debug("cdev_add failed!\n");
		goto free_cdev;
	}

	// init timer
	init_timer(&timer);
	timer.function = timer_func;
	timer.expires = jiffies + HZ/2;
	add_timer(&timer);

	return 0;

free_cdev:
    kobject_put(&driver_info->kobj);
    driver_info = NULL;	
free_devnum:
	unregister_chrdev_region(MKDEV(MAJORNUM,0), NUMDEVICES);
	return -1;
}

void fops_exit(void) {
	printk(KERN_INFO "Module blink: exit()\n");
	
	del_timer(&timer);
	outb(0x00,PORT);

	// remove char device from system
	if( driver_info ) cdev_del( driver_info );
    
	// free device number
	unregister_chrdev_region(MKDEV(MAJORNUM,0), NUMDEVICES);
}

module_init(fops_init);
module_exit(fops_exit);

/* FOPS Functions */
static int fops_open(struct inode* inode, struct file* file){

	if (is_open) return -EBUSY;
	
	is_open++;
	new_read++;
	try_module_get(THIS_MODULE);
	minor_num = iminor(inode);
	pr_debug("Module blink: device %s was opened from device with minor no %d\n", DEVNAME, minor_num);

	return 0;
}

static int fops_close(struct inode* inode, struct file* file){
		
	is_open = 0;
	new_read = 0;
	module_put(THIS_MODULE);
	pr_debug("Module blink: device %s was closed\n", DEVNAME);
	return 0;
}

static ssize_t fops_read(struct file* file, char* buffer, size_t len, loff_t* offset) {
	
	put_user(freq + 48, buffer++);
	put_user('\n', buffer);

	if (new_read) {
		new_read = 0;	
		return 2;
	}
	else return 0;

}

static ssize_t fops_write(struct file* file, const char* buf, size_t len, loff_t* offset){

	char ch;
	new_read++;
	if (len == 0) return 0;

	get_user(ch, buf);

	if (ch <= 57 && ch >= 48) {
		freq = ch - 48;
		pr_debug("Module blink: Freq set to %d Hz\n", freq);
	}
	else {
		pr_debug("Module blink: Freq must be in range 0..9!\n");
		return len;
	}

	return len;
}

static void timer_func(unsigned long ptr) {
	
	if (!freq) {
		outb(0x00,PORT);
		timer.expires = jiffies + HZ;
	}
	else {
		val = inb(PORT);
		outb(~val,PORT);
		timer.expires = jiffies + HZ/(2*freq);
	}
	add_timer(&timer);
}



