// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  fileoperations.c
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder
// Date:      25.10.2008
// *****************************************************************

#include <linux/version.h>	/* Linux Version */
#include <linux/module.h>	/* Makros and Defines */
#include <linux/fs.h>
#include <asm/uaccess.h>	/* put_user() */
#include <linux/cdev.h>

#include "ioctldefs.h"

#define MAJORNUM 60
#define NUMDEVICES 1
#define DEVNAME "myfirstdevice"
#define STDMSG "this is a boring standard message"

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Joachim Schroeder");
MODULE_DESCRIPTION("File Operations Example");


//#define COPY_MULTI_BYTES

/* Prototypes */
int fops_init(void);
void fops_exit(void);
static ssize_t fops_read(struct file *, char *, size_t, loff_t *);
static ssize_t fops_write(struct file *, const char *, size_t, loff_t *);
static int fops_open(struct inode *, struct file *);
static int fops_close(struct inode *, struct file *);
static int fops_ioctl( struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg );

/* Global Variables */
static int is_open = 0;
static char msg[MAXMSGLEN];
static char* p_msg;
static int msg_len = 0;

static struct cdev *driver_info = NULL;
static struct file_operations fops = {
	.owner		= 	THIS_MODULE,
	.read		=	fops_read,
	.write		=	fops_write,
	.ioctl		= 	fops_ioctl,
	.open		=	fops_open,
	.release	=	fops_close
};

int fops_init(void) {
	printk(KERN_INFO "Module fops: init()\n");

	// allocate device number
    if( register_chrdev_region( MKDEV(MAJORNUM,0), NUMDEVICES, DEVNAME ) ) {
        pr_debug("Device number 0x%x not available ...\n", MKDEV(MAJORNUM,0));
        return -EIO;
    }

	// allocate device
	driver_info = cdev_alloc();
    if( driver_info == NULL ) {
        pr_debug("cdev_alloc failed!\n");
        goto free_devnum;
    }

	// specify device structure (init) and register (add)
    kobject_set_name(&driver_info->kobj, DEVNAME );
    driver_info->owner = THIS_MODULE;
    cdev_init( driver_info, &fops );
    if( cdev_add( driver_info, MKDEV(MAJORNUM,0), NUMDEVICES) ) {
        pr_debug("cdev_add failed!\n");
        goto free_cdev;
    }
    return 0;

free_cdev:
    kobject_put(&driver_info->kobj);
    driver_info = NULL;	
free_devnum:
	unregister_chrdev_region(MKDEV(MAJORNUM,0), NUMDEVICES);
	return -1;
}

void fops_exit(void) {
	printk(KERN_INFO "Module fops: exit()\n");
	
	// remove char device from system
	if( driver_info ) cdev_del( driver_info );
    
	// free device number
	unregister_chrdev_region(MKDEV(MAJORNUM,0), NUMDEVICES);
}

module_init(fops_init);
module_exit(fops_exit);

/* FOPS Functions */
static int fops_open(struct inode* inode, struct file* file){
	
	if (is_open) return -EBUSY;
	
	is_open++;
	p_msg = msg;
	try_module_get(THIS_MODULE);
	pr_debug("Module fops: device %s was opened from device with minor no %d\n", DEVNAME, iminor(inode));
	return 0;
}

static int fops_close(struct inode* inode, struct file* file){
		
	is_open--;
	module_put(THIS_MODULE);
	pr_debug("Module fops: device %s was closed\n", DEVNAME);
	return 0;
}

#ifdef COPY_MULTI_BYTES
static ssize_t fops_read(struct file* file, char* buf, size_t len, loff_t* offset) {
	
	int num = 0;
	if (*msg == 0) {
		pr_debug("Module fops: Msg buffer is empty!\n");
		return 0;
	}
	
	/* write data to buf, bytes need to be transferred from KS to US */
	while(len && *p_msg && (p_msg - msg) < msg_len) {
		put_user(*(p_msg++), buf++);
		len--;
		num++;
	}
	pr_debug("Module fops: sent %d bytes to user space\n", num);
	return num;
}

static ssize_t fops_write(struct file* file, const char* buf, size_t len, loff_t* offset){
	
	int num = 0;
	p_msg = msg;	
	
	if (len > MAXMSGLEN) {
		pr_debug("Module fops: Msg buffer is too small, skipping %d bytes!\n", len - MAXMSGLEN);
	}
	
	/* read data from buf, bytes need to be transferred from US to KS */
	while(len && num < MAXMSGLEN) {		
		get_user(*(p_msg++), buf++);
		len--;
		num++;
	}
	msg_len = num;	
	p_msg = msg;
	pr_debug("Module fops: received %d bytes from user space\n", num);
	return num;
}

#else
static ssize_t fops_read(struct file* file, char* buf, size_t len, loff_t* offset) {
	
	int num = 0;
	if (*msg == 0) {
		pr_debug("Module fops: Msg buffer is empty!\n");
		return 0;
	}
	
	if (msg_len - (p_msg - msg) < len) {
		len = msg_len - (p_msg - msg);
		pr_debug("Module fops: not enough content, can read only %d bytes\n", len);		
	}

	num = len - copy_to_user(buf, p_msg, len);
	p_msg += num;
	pr_debug("Module fops: sent %d bytes to user space\n", num);
	return num;
}

static ssize_t fops_write(struct file* file, const char* buf, size_t len, loff_t* offset){
	
	int num = 0;
	if (len > MAXMSGLEN) {
		pr_debug("Module fops: Msg buffer is too small, skipping %d bytes!\n", len - MAXMSGLEN);
		len = MAXMSGLEN;
	}
	
	num = len - copy_from_user(msg, buf, len);
	msg_len = num;
	pr_debug("Module fops: received %d bytes from user space\n", num);
	return num;
}
#endif


static int fops_ioctl( struct inode *inode, struct file *file,
       unsigned int cmd, unsigned long arg )
{
	switch( cmd ) {                                        
		case IOCTL_CLRMSGBUF:
			p_msg = msg;
			msg_len = 0;
			pr_debug("Module fops: clearing message buffer\n");
			break;
    	case IOCTL_SETSTDMSG:
			strncpy(msg, STDMSG, strlen(STDMSG));
			p_msg = msg;
			msg_len = strlen(STDMSG);
			pr_debug("Module fops: setting message to standard with len %d\n", msg_len);
			break;
		default:
			pr_debug("unknown IOCTL 0x%x\n", cmd);
			return -EINVAL;
	}
	return 0;
}






