// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  modulereg.c
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder
// Date:      25.09.2008
// *****************************************************************

#include <linux/version.h>	/* Linux Version */
#include <linux/module.h>	/* Makros and Defines */
#include <linux/fs.h>
#include <linux/cdev.h>

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Joachim Schroeder");
MODULE_DESCRIPTION("Module Registration Argument Example");

#define MAJORNUM 60
#define NUMDEVICES 1
#define DEVNAME "myfirstdevice"

static struct cdev *driver_info = NULL;
static struct file_operations fops;

int modulereg_init(void) {
	printk(KERN_INFO "Module modulereg: init()\n");

	// allocate device number
	if( register_chrdev_region( MKDEV(MAJORNUM,0), NUMDEVICES, DEVNAME ) ) {
		pr_debug("Device number 0x%x not available ...\n", MKDEV(MAJORNUM,0));
		return -EIO;
	}

	// allocate device
	driver_info = cdev_alloc();
	if( driver_info == NULL ) {
		pr_debug("cdev_alloc failed!\n");
		goto free_devnum;
	}

	// specify device structure (init) and register (add)
	kobject_set_name(&driver_info->kobj, DEVNAME );
	driver_info->owner = THIS_MODULE;
	cdev_init( driver_info, &fops );
	if( cdev_add( driver_info, MKDEV(MAJORNUM,0), NUMDEVICES) ) {
		pr_debug("cdev_add failed!\n");
		goto free_cdev;
	}
	return 0;

free_cdev:
	kobject_put(&driver_info->kobj);
	driver_info = NULL;	
free_devnum:
	unregister_chrdev_region(MKDEV(MAJORNUM,0), NUMDEVICES);
	return -1;
}

void modulereg_exit(void) {
	printk(KERN_INFO "Module modulereg: exit()\n");
	
	// remove char device from system
	if( driver_info ) cdev_del( driver_info );
    
	// free device number
	unregister_chrdev_region(MKDEV(MAJORNUM,0), NUMDEVICES);
}

module_init(modulereg_init);
module_exit(modulereg_exit);


