// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************
// *****************************************************************
// Filename:  relaiscard.c
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder
// Date:      30.10.2008
// *****************************************************************

#include <linux/version.h>	/* Linux Version */
#include <linux/module.h>	/* Makros and Defines */
#include <linux/fs.h>
#include <asm/uaccess.h>	/* put_user() */
#include <linux/cdev.h>
#include <linux/sched.h>	/* struct task_struct *current */
#include <asm/fcntl.h>
#include <linux/delay.h>
#include <linux/spinlock.h>

#include "ioctldefs.h"

#define MAJORNUM 60
#define NUMDEVICES 8
#define DEVNAME "relaiscard"
#define SLEEPTIME 50 // in ms

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Joachim Schroeder");
MODULE_DESCRIPTION("File Access Example");

/* Prototypes */
int fops_init(void);
void fops_exit(void);
static ssize_t fops_read(struct file *, char *, size_t, loff_t *);
static ssize_t fops_write(struct file *, const char *, size_t, loff_t *);
static int fops_open(struct inode *, struct file *);
static int fops_close(struct inode *, struct file *);
static int fops_ioctl( struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg );

static int serial_init(void);
static int serial_sndCmd(unsigned char command, unsigned char addr, unsigned char data);
static int serial_rcvCmd(char* buf);
void serial_release(void);

/* Global Variables */
static char rel_state = 0x00;
static int is_open[8]= {0,0,0,0,0,0,0,0};
static int new_read[8] = {0,0,0,0,0,0,0,0};
static struct file *fd_p;
static mm_segment_t fs;
static char *device = NULL;
static spinlock_t lock = SPIN_LOCK_UNLOCKED;

static struct cdev *driver_info = NULL;
static struct file_operations fops = {
	.owner		= 	THIS_MODULE,
	.read		=	fops_read,
	.write		=	fops_write,
	.ioctl		= 	fops_ioctl,
	.open		=	fops_open,
	.release	=	fops_close
};

/* Parameters */
module_param(device, charp,0);
MODULE_PARM_DESC(device, "Serial device (/dev/ttyS0)");

int fops_init(void) {
	printk(KERN_INFO "Module relaiscard: init()\n");

	if (device == NULL) {
		printk(KERN_INFO "Module relaiscard: Parameter missing (e.g. device=""/dev/ttyS0"")\n");
		return -1;
	}

	// allocate device number
	if( register_chrdev_region( MKDEV(MAJORNUM,0), NUMDEVICES, DEVNAME ) ) {
		pr_debug("Device number 0x%02x not available ...\n", MKDEV(MAJORNUM,0));
		return -EIO;
	}

	// allocate device
	driver_info = cdev_alloc();
	if( driver_info == NULL ) {
		pr_debug("cdev_alloc failed!\n");
		goto free_devnum;
	}

	// specify device structure (init) and register (add)
	kobject_set_name(&driver_info->kobj, DEVNAME );
	driver_info->owner = THIS_MODULE;
	cdev_init( driver_info, &fops );
	if( cdev_add( driver_info, MKDEV(MAJORNUM,0), NUMDEVICES) ) {
		pr_debug("cdev_add failed!\n");
		goto free_cdev;
	}

	//init serial connection
	fs = get_fs();
	set_fs(KERNEL_DS);
   if( serial_init() < 0 ) {
        pr_debug("init_serial failed!\n");
        goto free_cdev;    
	}
	set_fs(fs);

	rel_state = 0x00;
	return 0;

free_cdev:
	set_fs(fs);
    kobject_put(&driver_info->kobj);
    driver_info = NULL;	
free_devnum:
	unregister_chrdev_region(MKDEV(MAJORNUM,0), NUMDEVICES);
	return -1;
}

void fops_exit(void) {
	printk(KERN_INFO "Module relaiscard: exit()\n");
	
	set_fs(KERNEL_DS);
		serial_release();
	set_fs(fs);

	// remove char device from system
	if( driver_info ) cdev_del( driver_info );
    
	// free device number
	unregister_chrdev_region(MKDEV(MAJORNUM,0), NUMDEVICES);
}

module_init(fops_init);
module_exit(fops_exit);

/* FOPS Functions */
static int fops_open(struct inode* inode, struct file* file){

	unsigned int minor = iminor(inode);
	if (minor < 0 || minor > 7) return -EBUSY; 
	if (is_open[minor]) return -EBUSY;
	
	is_open[minor]++;
	new_read[minor]++;
	try_module_get(THIS_MODULE);

	pr_debug("Module relaiscard: %s was opened from device with minor no %d\n", DEVNAME, minor);
	if (minor < 0 || minor > 7) {
		pr_debug("Module relaiscard: minor num must be in range 0..7\n");
		return -1;
	}

	return 0;
}

static int fops_close(struct inode* inode, struct file* file){
	
	unsigned int minor = iminor(inode);	
	is_open[minor] = 0;
	new_read[minor] = 0;
	module_put(THIS_MODULE);
	pr_debug("Module relaiscard: %s was closed from device with minor no %d\n", DEVNAME, minor);
	return 0;
}

static ssize_t fops_read(struct file* file, char* rbuf, size_t len, loff_t* offset) {
	
	char ch;
	char buf[4];
	unsigned int minor = MINOR(file->f_dentry->d_inode->i_rdev);
	pr_debug("Module relaiscard: read access from device with minor no %d\n", minor);

	set_fs(KERNEL_DS);
		if (serial_sndCmd(2,1,0) != 0) goto read_fail;
		if (serial_rcvCmd(buf) != 0) goto read_fail;
	set_fs(fs);

	if (buf[0] == (char)0xFD) rel_state = buf[2];
	else goto read_fail;	
	
	if ((rel_state >> minor) % 2) ch = '1';
	else ch = '0';
	put_user(ch, rbuf++);
	put_user('\n', rbuf);

	if (new_read[minor]) {
		new_read[minor] = 0;	
		return 2;
	}
	else {
		new_read[minor] = 1;
		return 0;
	}

read_fail:
	pr_debug("Module relaiscard: read() failed\n");
	set_fs(fs);
	return -1;
}

static ssize_t fops_write(struct file* file, const char* wbuf, size_t len, loff_t* offset){

	char ch;
	char buf[4];
	unsigned int minor = MINOR(file->f_dentry->d_inode->i_rdev);
	pr_debug("Module relaiscard: write access from device with minor no %d\n", minor);

	new_read[minor]++;
	if (len == 0) return 0;

	get_user(ch, wbuf);
	
	if (ch == '1') {
		rel_state |= (1 << minor);
	}
	else if (ch == '0') {
		rel_state &= ~(1 << minor);
	}
	else {
		pr_debug("Module relaiscard: Cannot handle character 0x%0x!\n", ch);
		return len;
	}

	set_fs(KERNEL_DS);
		if (serial_sndCmd(3,1,rel_state) != 0) pr_debug("Module relaiscard: Setting to state 0x%0x failed (snd)\n", rel_state);
		if (serial_rcvCmd(buf) != 0) pr_debug("Module relaiscard: Setting to state 0x%0x failed (rcv)\n", rel_state);
	set_fs(fs);

	return len;
}

static int fops_ioctl( struct inode *inode, struct file *file,
       unsigned int cmd, unsigned long arg )
{
	char buf[4];	

    switch( cmd ) {                                        
    case IOCTL_DISABLE_ALL:
		rel_state = 0x00;
		pr_debug("Module relaiscard: ioctl() disable all\n");
		set_fs(KERNEL_DS);
			if (serial_sndCmd(3,1,rel_state) != 0) pr_debug("Module relaiscard: Setting to state 0x%0x failed (snd)\n", rel_state);
			if (serial_rcvCmd(buf) != 0) pr_debug("Module relaiscard: Setting to state 0x%0x failed (rcv)\n", rel_state);
		set_fs(fs);
		break;
	case IOCTL_ENABLE_ALL:
		rel_state = 0xff;
		pr_debug("Module relaiscard: ioctl() enable all\n");
		set_fs(KERNEL_DS);
			if (serial_sndCmd(3,1,rel_state) != 0) pr_debug("Module relaiscard: Setting to state 0x%0x failed (snd)\n", rel_state);
			if (serial_rcvCmd(buf) != 0) pr_debug("Module relaiscard: Setting to state 0x%0x failed (rcv)\n", rel_state);
		set_fs(fs);
		break;
    default:
       pr_debug("unknown IOCTL 0x%x\n", cmd);
       return -EINVAL;
    }
    return 0;
}


/* Serial Helper Functions */
static int serial_init() {
	
	char buf[4];
	pr_debug("Module relaiscard: serial_init()\n");

	fd_p = filp_open(device, O_RDWR, 0);
	if (fd_p < 0) {
		pr_debug("Module relaiscard: Error opening %s\n",device);
		return -1;
	}

	if (serial_sndCmd(1,1,0) != 0) return -1;
	if (serial_rcvCmd(buf) != 0) return -1;

	spin_lock(&lock);
		if (fd_p->f_op->read(fd_p,buf,4,0) != 4) return -1;	// don't know why need to do this..
	spin_unlock(&lock);

	return 0;
}

int serial_sndCmd(unsigned char command, unsigned char addr, unsigned char data) {
	
	char buf[4];
	int ret;
	buf[0] = command;
	buf[1] = addr;
	buf[2] = data;
	buf[3] = buf[0]^buf[1]^buf[2];	// calculation of the checksum

	pr_debug("Module relaiscard: -> sncmd: 0x%0x 0x%0x 0x%0x 0x%0x\n", buf[0], buf[1], buf[2], buf[3]);

	spin_lock(&lock);
		ret = fd_p->f_op->write(fd_p,buf,4,0);
		mdelay(SLEEPTIME);	// short pause; card misses some messages otherwise
	spin_unlock(&lock);

	if ( ret != 4) {
		pr_debug("send failed %d\n",ret);		
		return -1;
	}
	return 0; 
}

int serial_rcvCmd(char* rbuf) {

	int ret;

	spin_lock(&lock);
		ret = fd_p->f_op->read(fd_p,rbuf,4,0);
	spin_unlock(&lock);
	if ( ret != 4) return -1;

	pr_debug("Module relaiscard: -> recv: 0x%0x 0x%0x 0x%0x 0x%0x\n", rbuf[0], rbuf[1], rbuf[2], rbuf[3]);
	
	if ( (rbuf[0]^rbuf[1]^rbuf[2]) == rbuf[3]) return 0;
	else {
		pr_debug("Module relaiscard: serial_rcvCmd(): Checksum wrong!\n");
		return(-1);
	}
}

void serial_release() {

	pr_debug("Module relaiscard: serial_release()\n");

	if (filp_close(fd_p,NULL) != 0) pr_debug("Module relaiscard: Error closing %s\n",device);

}





