// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  main.cpp
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder
// Date:      03.11.2008
// *****************************************************************

#include <cstdlib>					// atoi()

#include "../udp_qt_server/Datagrams.h" 
#include "tools/PracticalSocket.h" 
#include "tools/Thread.h" 
#include "tools/Mutex.h" 

#define CYCLETIME 200000	// Send data every 200 ms
#define STEP_DELTA 5			// steps per cycl towards goal pos.
#define SIGN(a) ((a<0) ? -1 : 1)

class ReceiveThread : public Thread {

public:
	ReceiveThread(DatagramControl& control, Mutex& mutex_control, unsigned short listen_port)
		: data_control(control), mutex_data_control(mutex_control) {
		rec_sock.setLocalPort(listen_port);
	}
	
	virtual void run() {
	
		string source_address;             // Source address
		unsigned short source_port;        // Pource port
	 
		cout << "Receive Thread started" << endl;
		while(1) {
			
			int rec_size = rec_sock.recvFrom(&temp_data_control, source_address, source_port);  
			try {
				cout << "Received new Target Position: " << temp_data_control.data.target_pos << " from " << source_address << ":" << source_port<< endl;
				MutexLocker ml(mutex_data_control);
				data_control = temp_data_control;
			}
			catch (SocketException) {
				cout << "Received exception " << endl;
				sleep(1);
			}
		}
	}
	
	~ReceiveThread() {
		rec_sock.disconnect();
	}
	
private:
	UDPSocket rec_sock;
	DatagramControl temp_data_control;
	DatagramControl& data_control;
	Mutex& mutex_data_control;
		
};


int main(int argc, char *argv[]) {

	if (argc != 4) {   // Test for correct number of arguments
		cout << "Usage: " << argv[0] << " <Server> <Server Port> <Receive Port>\n";
 		exit(1);
	}

	string serv_address = argv[1]; // server address
	unsigned short serv_port = atoi(argv[2]);
	unsigned short listen_port = atoi(argv[3]);
	
	UDPSocket send_sock;
	DatagramMeasure mes;
	mes.data.current_pos = 0;
	Mutex mutex_control;
	DatagramControl control;
	control.data.target_pos = 0;

	ReceiveThread th(control, mutex_control, listen_port);
	th.start();
	
	while(1) {
	
		// compute new value
		mutex_control.lock();
		int diff = control.data.target_pos - mes.data.current_pos;
		
		if ( abs(diff) < STEP_DELTA) {
			mes.data.current_pos = control.data.target_pos;
		}
		else {
			mes.data.current_pos += SIGN(diff) * STEP_DELTA;
		}
		mutex_control.unlock();
	
		
		// Send the current position to the server
		send_sock.sendTo(&mes, serv_address, serv_port);
		cout << "Sending message no " << mes.getCount() << endl;
		
		usleep(CYCLETIME);
	}
	send_sock.disconnect();
	
	th.wait();
	return 0;
}
