// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  main.cpp
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder
// Date:      27.04.2008
// ***************************************************************** 

#include "../udp_qtwidget_server/Datagrams.h"
#include "tools/PracticalSocket.h"
#include "tools/Thread.h"
#include "tools/Mutex.h"
#include "iic/Bus.h"
#include "iic/Stepper.h"

#define CYCLETIME 200000	// Send every 200 ms
#define STEP_DELTA 5			// number of steps to move towards target position in each cycle
#define SIGN(a) ((a<0) ? -1 : 1)
#define MAX_REFERENCE_RUN 20000
#define SWITCH_REFERENCE_DIST 10

class ReceiveThread : public Thread {

public:
	ReceiveThread(DatagramControl& control, Mutex& m_control, DatagramParam& param, Mutex& m_param, unsigned short listen_port)
		: 	data_control(control), 
			m_data_control(m_control),
			data_param(param),
			m_data_param(m_param)
	{
		rec_sock.setLocalPort(listen_port);
	}
	
	void run() {
	
		string sourceAddress;             // Address of datagram source
		unsigned short sourcePort;        // Port of datagram source
	 
		cout << "Receive Thread started" << endl;
		while(1) {
			
			try {
				// do not take data from socket until you know what it is
				int rec_size = rec_sock.recvFromPeek(&header, sizeof(datagram_header_t), sourceAddress, sourcePort); 
				AbstractDatagram::ntohHeader(&header); 
		
				if (header.id == CONTROL) {
					MutexLocker ml(m_data_control);
					int rec_size = rec_sock.recvFrom(&data_control, sourceAddress, sourcePort);  
				}
				else if (header.id == PARAM) {
					MutexLocker ml(m_data_param);
					
					int rec_size = rec_sock.recvFrom(&data_param, sourceAddress, sourcePort);  				
				}
			}
			catch (SocketException) {
				cout << "Received exception " << endl;
				sleep(1);
			}
		}
	}
	
	~ReceiveThread() {
		rec_sock.disconnect();
	}
	
private:
	UDPSocket rec_sock;
	DatagramControl& data_control;
	Mutex& m_data_control;
	DatagramParam& data_param;
	Mutex& m_data_param;
	datagram_header_t header;	
};


int main(int argc, char *argv[]) {

	if (argc != 4) {   // Test for correct number of arguments
		cout << "Usage: " << argv[0] << " <Server> <Server Port> <Receive Port>\n";
 		exit(1);
	}

	string serv_address = argv[1]; // server address
	unsigned short serv_port = atoi(argv[2]);
  	unsigned short listen_port = atoi(argv[3]);
	
	UDPSocket send_sock;
	DatagramMeasure mes_data;
	DatagramParam param_data;
	DatagramControl control_data;
	Mutex m_param_data;
	Mutex m_control_data;

	motor_params basic_params;
	
	control_data.data.target_pos = 0;
	control_data.data.new_target_pos = 0;
	control_data.data.goto_ref_pos = 0;

	ReceiveThread th(control_data, m_control_data, param_data, m_param_data, listen_port);
	th.start();
	
	//IICBus bus("/dev/i2c-0",0);
	IICBus bus("/dev/iowarrior",1,"000023DD");
	IICStepper stepper(bus,0x60, "Stepper Modul");
	
	bus.printModules();
	bus.autodetect();
		
	bus.init();
		
	basic_params.i_run			= 0x0B; // 400mA peak current
	basic_params.i_hold			= 0x03;	// 100mA hold current
	basic_params.v_max			= 0x02;
	basic_params.v_min			= 0x01;	// 1/32 v_max		
	basic_params.shaft			= 1;
	basic_params.step_mode	= 0x03; // 1/16 stepping mode
	basic_params.acc				= 0x01;
	basic_params.acc_shape	= 0x00; // use acc param

	param_data.data.new_data = 0;
	
	// complete init procedure with drive to reference switch
	stepper.referenceInit(basic_params, basic_params.v_max, basic_params.v_min, MAX_REFERENCE_RUN, SWITCH_REFERENCE_DIST);
	
	while(1) {
	
		//process control data
		m_control_data.lock();
		
		if (control_data.data.goto_ref_pos) {
			cout << "Going to reference position " << endl;
			stepper.referenceInit(basic_params, basic_params.v_max, basic_params.v_min, MAX_REFERENCE_RUN, SWITCH_REFERENCE_DIST);
			control_data.data.goto_ref_pos = 0;
		}
		
		if (control_data.data.new_target_pos) {
			cout << "Going to target position " << control_data.data.target_pos << endl;
			stepper.setPosition(control_data.data.target_pos);
			control_data.data.new_target_pos = 0;
		}
		
		m_control_data.unlock();
		
		//process param data
		m_param_data.lock();
		
		if (param_data.data.new_data) {
			cout << "Setting new Parameters " << endl;
			stepper.setMotorParam( param_data.data.param, 0 );
			param_data.data.new_data = 0;
		}
		m_param_data.unlock();
	
		mes_data.data.status = stepper.getMotorStatus();
		stepper.getFullStatus2();
		mes_data.data.actual_pos = stepper.getActualPos();
		
		// Send the current position and status info to the server
		send_sock.sendTo(&mes_data, serv_address, serv_port);
		
		usleep(CYCLETIME);
	}
	send_sock.disconnect();
	
	th.wait();
	return 0;

}
