// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  ControlPanel.cpp
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder
// Date:      04.11.2008
// *****************************************************************

#include <QWidget>
#include <QApplication>
#include <QTimer>
#include <iostream>
#include "ControlPanel.h"

ControlPanel::ControlPanel(char *argv [], QWidget *parent)
	: QWidget(parent)
{
	client_address = QHostAddress(argv[1]);
	client_port = atoi(argv[2]);
	server_port = atoi(argv[3]);
		
	setupUi(this);

	button_quit->setFont(QFont("Times", 18, QFont::Bold));
	lcd_mesg_rec->setNumDigits(3);
	lcd_mesg_send->setNumDigits(3);
	
	udpSocketSend = new QUdpSocket(this);
	udpSocketReceive = new QUdpSocket(this);
	udpSocketReceive->bind(server_port);
	timer = new QTimer(this);

	slider_target_pos->setMinimum(0);
	slider_target_pos->setMaximum(10000);
	progress_actual_pos->setMinimum(0);
	progress_actual_pos->setMaximum(10000);
  
	connect(button_quit, SIGNAL(clicked()), qApp, SLOT(quit()));
	connect(timer, SIGNAL(timeout()), this, SLOT(noConn()));
	connect(slider_target_pos, SIGNAL(valueChanged(int)), lcd_target_pos, SLOT(display(int)));
 	connect(button_send_params, SIGNAL(clicked()), this, SLOT(broadcastDatagramParam()));
 	connect(button_goto_ref_pos, SIGNAL(clicked()), this, SLOT(sendGotoRefpos()));
 	connect(button_send_target_pos, SIGNAL(clicked()), this, SLOT(sendNewTargetpos()));
 	connect(udpSocketReceive, SIGNAL(readyRead()), this, SLOT(processPendingDatagrams()));
	timer->start(1000);
	
	// set default param values
	spin_max_current->setRange(0,15);
	spin_max_current->setValue(0x0B); 	// 400mA peak current
	spin_hold_current->setRange(0,15);
	spin_hold_current->setValue(0x03);	// 100mA hold current
	spin_max_speed->setRange(0,15);
	spin_max_speed->setValue(0x04);
	spin_min_speed->setRange(0,15);
	spin_min_speed->setValue(0x01);		// 1/32 v_max		
	spin_mode->setRange(0,15);
	spin_mode->setValue(0x03); 			// 1/16 stepping mode
	spin_acc->setRange(0,15);
	spin_acc->setValue(0x02);

	num_mesg_rec = 0;
	num_mesg_send = 0;
}

ControlPanel::~ControlPanel() {}

void ControlPanel::noConn() {
	label_status->setText("Not connected");
}

void ControlPanel::broadcastDatagramParam() {

	param_data.data.new_data = 1;

	param_data.data.param.i_run = spin_max_current->value();
	param_data.data.param.i_hold = spin_hold_current->value();
	param_data.data.param.v_max = spin_max_speed->value();
	param_data.data.param.v_min = spin_min_speed->value();	
	param_data.data.param.shaft = 1;
	param_data.data.param.step_mode = spin_mode->value();
	param_data.data.param.acc = spin_acc->value();
	param_data.data.param.acc_shape = 0;

	param_data.incCount();
	param_data.hton();
	qint64 res = udpSocketSend->writeDatagram(param_data.getData(), param_data.getSize(), client_address, client_port);
    param_data.ntoh();                          
	if (res < 0) {
		std::cout << "Param Data could not be written" << std::endl;
	}
	else {
		lcd_mesg_send->display(++num_mesg_send);
	}
}

void ControlPanel::sendNewTargetpos() {		

	control_data.data.goto_ref_pos = 0;
	control_data.data.new_target_pos = 1;	
	control_data.data.target_pos = slider_target_pos->value();

	broadcastDatagramControl();
}

void ControlPanel::sendGotoRefpos() {

	control_data.data.goto_ref_pos = 1;
	control_data.data.new_target_pos = 0;		

	broadcastDatagramControl();
}

void ControlPanel::broadcastDatagramControl() {

	control_data.incCount();
	control_data.hton();
	qint64 res = udpSocketSend->writeDatagram(control_data.getData(), control_data.getSize(), client_address, client_port);
    control_data.ntoh();                          
	if (res < 0) {
		std::cout << "Control Data could not be written" << std::endl;
	}
	else {
		lcd_mesg_send->display(++num_mesg_send);
	}
}

void ControlPanel::processPendingDatagrams()
{
	QHostAddress sender;
	quint16 senderport;

    while (udpSocketReceive->hasPendingDatagrams()) {

		udpSocketReceive->readDatagram(mes_data.getData(), mes_data.getSize(), &sender, &senderport);
		mes_data.ntoh();
		timer->start(1000);
		label_status->setText(sender.toString());
		lcd_mesg_rec->display(++num_mesg_rec);

    	led_temp_warn->setValue(true);
		if (mes_data.data.status.temp_warn) led_temp_warn->setColor(Qt::red);
		else led_temp_warn->setColor(Qt::green);

    	led_temp_shutdown->setValue(true);
		if (mes_data.data.status.temp_down) led_temp_shutdown->setColor(Qt::red);
		else led_temp_shutdown->setColor(Qt::green);

    	led_overcurrent->setValue(true);
		if (mes_data.data.status.over_current_a || mes_data.data.status.over_current_b) led_overcurrent->setColor(Qt::red);
		else led_overcurrent->setColor(Qt::green);

    	led_undervoltage->setValue(true);
		if (mes_data.data.status.uv2) led_undervoltage->setColor(Qt::red);
		else led_undervoltage->setColor(Qt::green);

    	led_step_loss->setValue(true);
		if (mes_data.data.status.step_loss) led_step_loss->setColor(Qt::red);
		else led_step_loss->setColor(Qt::green);

    	led_motion->setValue(true);
		if (mes_data.data.status.motion) led_motion->setColor(Qt::red);
		else led_motion->setColor(Qt::green);

    	led_ext_switch->setValue(true);
		if (mes_data.data.status.ext_switch) led_ext_switch->setColor(Qt::red);
		else led_ext_switch->setColor(Qt::green);

		lcd_temp->display(mes_data.data.status.temp_info);
		lcd_actual_pos->display(mes_data.data.actual_pos);
      	progress_actual_pos->setValue(mes_data.data.actual_pos); 
	}
}
 
 
 
 
 
 
 
 
