// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  Queue.h
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Daniel Jagszent
// Date:      26.04.2008
// *****************************************************************


#ifndef QUEUE_H_
#define QUEUE_H_

#include "tools/Mutex.h"
#include "tools/WaitCondition.h"
#include <iostream>
#include <deque>
#include <string>
#include <string.h>
#include <unistd.h> // usleep
#include <cstdlib> // rand / RAND_MAX


template<class ITEM>
class Queue {
public:
  typedef ITEM item_t;

  explicit Queue(unsigned int max_size = 0):
    cond_not_empty(mutex), cond_not_full(mutex), cond_all_items_finished(mutex),
    max_size(max_size), unfinished_items(0) {};

  item_t get()
  {
    MutexLocker ml(mutex);

    // wait until an item is available
    while (items.empty()) {
      cond_not_empty.wait(); // automatically unlocks the mutex while waiting
    }

    // get the item
    item_t an_item = items.front();
    items.pop_front();

    // and signal any producer waiting for putting new items in queue
    cond_not_full.signalOne();
    return an_item;
  }

  void put(const item_t& item)
  {
    MutexLocker ml(mutex);

    // wait until the queue has free capacity
    while (max_size > 0 && items.size() > max_size) {
      cond_not_full.wait(); // automatically unlocks the mutex while waiting
    }

    // put item in queue
    items.push_back(item);

    // keep track of item for wait()
    ++unfinished_items;

    // wake one consumer
    cond_not_empty.signalOne();
  }

  void item_finished()
  {
    MutexLocker ml(mutex);
    --unfinished_items;
    if (unfinished_items < 0) {
      // error!
      std::cout << "error! item_finished() called more times than put()" << std::endl;
    }

    // unfinished_items == 0 when queue is empty AND all consumers called item_finished() for the items they got
    if (unfinished_items <= 0) {
      // wake up all waiting threads
      cond_all_items_finished.signalAll();
    }
  }

  void wait()
  {
    MutexLocker ml(mutex);
    // wait until all consumers reported that their processing is done
    while (unfinished_items > 0) {
      cond_all_items_finished.wait(); // automatically unlocks the mutex while waiting
    }
  }
protected:
  std::deque< item_t > items;
  Mutex mutex;
  WaitCondition cond_not_empty, cond_not_full, cond_all_items_finished;
  size_t max_size;
  int unfinished_items;
};


#endif /* QUEUE_H_ */
