// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  main.cpp
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder, Daniel Jagszent
// Date:      26.04.2008
// *****************************************************************

#include "tools/Thread.h"
#include "Queue.h"

Mutex cout_mutex;

class Producer: public Thread
{
public:
  Producer(const std::string& name, Queue<int>* queue): name(name), queue(queue){};
  void run() {
    for (int i = 0; i<10; ++i) {
      {
        MutexLocker ml(cout_mutex);
        std::cout << name << " makes " << i << " available." << std::endl;
      }
      queue->put(i);
      usleep(100+(int) (500.0*rand()/(RAND_MAX+1.0)));
    }
  };
  std::string name;
protected:
  Queue<int>* queue;
};

class Consumer: public Thread
{
public:
  Consumer(const std::string& name, Queue<int>* queue): name(name), queue(queue){};
  void run() {
    while (true) {
      int i = queue->get();
      {
        MutexLocker ml(cout_mutex);
        std::cout << name << " got " << i << "." << std::endl;
      }
      usleep(100+(int) (1000.0*rand()/(RAND_MAX+1.0)));
      queue->item_finished();
    }
  };
  std::string name;
protected:
  Queue<int>* queue;
};


typedef std::deque < Producer* > ProducerList;
typedef std::deque < Consumer* > ConsumerList;

int main() {

  // create a queue with 20 slots
  Queue<int> queue(20);

  ProducerList producers;
  ConsumerList consumers;
  char thread_name[100] = {0};

  // create producers and consumers
  for (int i=0; i < 10; ++i)
  {
    sprintf(thread_name, "Producer %d", i);
    producers.push_back(new Producer(thread_name, &queue));
  }
  for (int i=0; i < 10; ++i)
  {
    sprintf(thread_name, "Consumer %d", i);
    consumers.push_back(new Consumer(thread_name, &queue));
  }

  // start them
  for (ConsumerList::iterator i = consumers.begin(); i != consumers.end(); ++i)
  {
    (*i)->start();
  }
  for (ProducerList::iterator i = producers.begin(); i != producers.end(); ++i)
  {
    (*i)->start();
  }

  // and wait until all work is done
  for (ProducerList::iterator i = producers.begin(); i != producers.end(); ++i)
  {
    {
      MutexLocker ml(cout_mutex);
      std::cout << "Wait for " << (*i)->name << " to finish" << std::endl;
    }
    (*i)->wait();
  }
  {
    MutexLocker ml(cout_mutex);
    std::cout << "Wait for Queue to be empty (=all consumers are finished)" << std::endl;
  }
  queue.wait();

  return 0;

}


