// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  ServoThread.cpp
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder, Daniel Jagszent
// Date:      26.04.2008
// *****************************************************************


#include "ServoThread.h"
#include "tools/timing_functions.h"
#include "io.h"
#include <string.h>


inline long long get_peak_duration(const int& i) {
  if (i / NUM_STEPS % 2) {
    return MIN_PEAK + PEAK_STEP * (i % NUM_STEPS);
  } else {
    return MAX_PEAK - PEAK_STEP * (i % NUM_STEPS);
  }
}

ServoThread::ServoThread() :
    PeriodicThread(CYCLE), m_i(0), m_servo_position(0),
  m_measurements(&m_measurements1[0]), m_measurements_index(-1),
  m_stats_finished(m_stats_mutex)
#ifdef __XENO__
  ,m_xeno_wait(m_xeno_mutex)
#endif
{
}

ServoThread::~ServoThread() {
}

void ServoThread::do_turn() {
  io_set_high(); // begin of peak
  timespec start_period, timeout, now;
  long long servo_position = m_servo_position;
  getCurrentTime(start_period);
  timeout = start_period;
  if (servo_position == 0) {
    timeout.tv_nsec += get_peak_duration(m_i);
  } else {
    timeout.tv_nsec += servo_position;
  }
  normalizeTimespec(timeout);
  // busy waiting until the peak is over
  // on systems with a clock resolution of 10ms or worst,
  // we could not reach the desired accuracy of 1-2ms with nanosleep
  // or another blocking mechanism
  while (getCurrentTime(now) < timeout)
    ;
  io_set_low(); // end of peak
  ++m_i;
  {
    MutexLocker m(m_stats_mutex);
    long long start = getNanoseconds(start_period);
    if (m_measurements_index > -1) {
      m_measurements[m_measurements_index] = start - m_last_period_start;
      ++m_measurements_index;
      if (m_measurements_index == NUM_MEASUREMENTS) {
          m_measurements_index = -1;     // deactivate measurement
          m_stats_finished.signalAll(); // signal other threads that a measurement is finished
      }
    }
    m_last_period_start = start;
  }
}

void ServoThread::start_stats(){
  MutexLocker m(m_stats_mutex);
  m_measurements_index = 0;
}


void ServoThread::set_servo_position(int newPosition) {
  if (newPosition > -1 && newPosition < 10) {
    m_servo_position = MIN_PEAK + ((MAX_PEAK - MIN_PEAK) / 9 * (newPosition % 10));
  } else {
    m_servo_position = 0;
  }
}
int ServoThread::get_servo_position()
{
  long long servo_position = m_servo_position;
  if (servo_position) {
    return (m_servo_position - MIN_PEAK) / ((MAX_PEAK - MIN_PEAK) / 10);
  } else {
    return (get_peak_duration(m_i) - MIN_PEAK) / ((MAX_PEAK - MIN_PEAK) / 10);
  }
}

ServoThread::measurment_t* ServoThread::get_stats() {
  measurment_t* measurements;
  m_stats_mutex.lock();
  m_stats_finished.wait();
  measurements = m_measurements;
  if (m_measurements == &m_measurements1[0]) {
    m_measurements = &m_measurements2[0];
  } else {
    m_measurements = &m_measurements1[0];
  }
  m_measurements_index = -1;
  m_stats_mutex.unlock();

  measurment_t* measurements_copy = new measurment_t[NUM_MEASUREMENTS];
  memcpy(measurements_copy, measurements, NUM_MEASUREMENTS * sizeof(measurment_t));

  return measurements_copy;
}

