// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  UiThread.cpp
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder, Daniel Jagszent
// Date:      26.04.2008
// *****************************************************************


#include "UiThread.h"
#include "tools/Thread.h"
#include "io.h"

#include <iostream>
#include <iomanip>
#include <fstream>
#include <stdlib.h>
#include <signal.h>
#include <stdio.h>
#include <unistd.h>
#include <math.h>

using namespace std;

UiThread::UiThread(ServoThread* ioThread) :
  Thread(), m_ioThread(ioThread) {
  print_help();
}


void UiThread::run() {
  char ch = 0;
  ServoThread::measurment_t* measurements;
  while (m_ioThread->isRunning()) {
    cin >> ch;
    switch (ch) {
    case 'Q':
    case 'q':
      cout << "Terminating..." << endl;
      m_ioThread->terminate();
      return;
      break;
    default:
      cout << "Unknown command!" << endl;
    case '?':
    case 'H':
    case 'h':
      print_help();
      break;
    case 'P':
    case 'p':
      cout << "Pausing Servo Thread..." << endl;
      m_ioThread->pause();
      break;
    case 'R':
    case 'r':
      cout << "Resuming Servo Thread..." << endl;
      m_ioThread->resume();
      break;
    case 'T':
    case 't':
      cout << "Start Timing..." << endl;
      m_ioThread->resume();
      m_ioThread->start_stats();
      measurements = m_ioThread->get_stats();
      cout << "Timing finished... (written to timing.data)" << endl;
      process_timing(measurements);
      delete[] measurements;
      measurements = 0;
      break;
    case 'A':
    case 'a':
      cout << "Automatic Servo movement..." << endl;
      m_ioThread->set_servo_position(-1);
      break;
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
      cout << "Set servo position to " << ch << "..." << endl;
      m_ioThread->set_servo_position((int)(ch-'0'));
      break;
    }
  }
}

void UiThread::print_help(){
  cout << "\n\n"
  " Q     -  Quit application\n"
  " P     -  Pause Servo Thread\n"
  " R     -  Resume Servo Thread\n"
  " T     -  Start a timing (written to timing.data)\n"
  " 0-9   -  Set the servo position (0=leftmost, 9=rightmost)\n"
  " A     -  Automatic servo movement\n"
  "\n"
  << endl;
}

void UiThread::process_timing(ServoThread::measurment_t* measurement) {
  try {
    ofstream plot_file("timing.data");
    ServoThread::measurment_t min, max;
    double avg;
    for (int i=0; i < ServoThread::NUM_MEASUREMENTS; ++i) {
      ServoThread::measurment_t value = measurement[i];
      ServoThread::measurment_t jitter = value - CYCLE;
      plot_file << value << " " << jitter << "\n";
      jitter = abs(jitter);
      if (i==0) {
        min = jitter; max = jitter; avg = jitter;
      } else {
        if (min > jitter) min = jitter;
        if (max < jitter) max = jitter;
        avg = (avg + jitter) / 2.0;
      }
    }
    cout << fixed << setprecision(3) << right <<
        "Timinig result:\n"
        "avg:" << setw(9)<< (avg/1000.0) << "us\n"
        "min:" << setw(9)<< (min/1000.0) << "us\n"
        "max:" << setw(9)<< (max/1000.0) << "us\n" << endl;
  } catch(...) {
    cout << "Error while writing timing data. Do you have permission to write to the file timinig.data?" << endl;
  }
}

