// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  GPSReceiver.cpp
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Stephan Riedel, stephan-riedel@gmx.de
// Date:      12.07.2008
// *****************************************************************

#include "GPSReceiver.h"

const double GPSReceiver::deg2rad = 3.141592653 / 180.0;
//Ellipsoid( 23, "WGS-84", 6378137, 0.00669438)
const double GPSReceiver::a = 6378137  ;
const double GPSReceiver::eccSquared = 0.00669438;
const double GPSReceiver::k0 = 0.9996;

GPSReceiver::GPSReceiver(char* device_name) {

	used_satellites = 0;
	satellites = 0;
	latitude = 0;
	longitude = 0;
	altitude = 0;
	speed = 0;
	time = 0;
	utm_x = 0;
	utm_y = 0;

	session.gpsdata.baudrate = 4800;
	session.gpsdata.parity = 0;
	session.gpsdata.stopbits = 1;

	gpsd_init(&session,&context,device_name);
	printf("Baud %d \n",session.gpsdata.baudrate);

	if (gpsd_activate(&session,true) < 0) {
		printf("GPS device nonexistent or can't be read");
		exit(1);
	}
	else printf("Session activated\n");

}

GPSReceiver::~GPSReceiver() {

	gpsd_wrap(&session);
}

int GPSReceiver::updateData() {
	
	gps_mask_t mask = gpsd_poll(&session);

	if(!isnan(session.gpsdata.satellites_used))			used_satellites = session.gpsdata.satellites_used;
	if(!isnan(session.gpsdata.satellites))				satellites = session.gpsdata.satellites;
	if(!isnan(session.gpsdata.fix.latitude))			latitude = session.gpsdata.fix.latitude;
	if(!isnan(session.gpsdata.fix.longitude))			longitude = session.gpsdata.fix.longitude;
	if(!isnan(session.gpsdata.fix.altitude))			altitude = session.gpsdata.fix.altitude;
	if(!isnan(session.gpsdata.fix.speed*KNOTS_TO_KPH))	speed = session.gpsdata.fix.speed*KNOTS_TO_KPH;
	if(!isnan(session.gpsdata.fix.time))				time = session.gpsdata.fix.time;
	
	char utm_zone[4];

	if (!isnan(latitude) && !isnan(longitude)) {  // don't make the conversion if lat or long ist NAN
		LLtoUTM(latitude, longitude, utm_x, utm_y, utm_zone);
	}

	return mask;
}

int GPSReceiver::printData() {

	printf("SATS:(%i/%i) LAT:%f LON:%f ALT:%.2f SPEED:%.2f TIME:%.0f UTM_X:%.2f UTM_Y:%.2f\n",
		used_satellites, satellites, latitude, longitude, altitude, speed, time, utm_x, utm_y);
	return 0;
}

int GPSReceiver::getSatellites() {
	if (isnan(satellites)) return -1;	
	else return satellites;
}

double GPSReceiver::getSpeed() {
	if (isnan(speed)) return -1;
	else return speed;
}

double GPSReceiver::getTime() {
	if (isnan(time)) return -1;
	else return time;
}

double GPSReceiver::getUTMX() {
	if (isnan(utm_x)) return -1;
	return utm_x;
}

double GPSReceiver::getUTMY() {
	if (isnan(utm_y)) return -1;
	return utm_y;
}

double GPSReceiver::getAltitude() {
	if (isnan(altitude)) return -1;
	return altitude;
}


void GPSReceiver::LLtoUTM(const double Lat, const double Long, double &UTMNorthing, double &UTMEasting, char* UTMZone) {

	double LongOrigin;
	double eccPrimeSquared;
	double N, T, C, A, M;

	//Make sure the longitude is between -180.00 .. 179.9
	double LongTemp = (Long+180)-int((Long+180)/360)*360-180; // -180.00 .. 179.9;

	double LatRad = Lat*deg2rad;
	double LongRad = LongTemp*deg2rad;
	double LongOriginRad;
	int    ZoneNumber;

	ZoneNumber = int((LongTemp + 180)/6) + 1;

	if( Lat >= 56.0 && Lat < 64.0 && LongTemp >= 3.0 && LongTemp < 12.0 ) ZoneNumber = 32;

	// Special zones for Svalbard
	if( Lat >= 72.0 && Lat < 84.0 ) {
		if(      LongTemp >= 0.0  && LongTemp <  9.0 ) ZoneNumber = 31;
		else if( LongTemp >= 9.0  && LongTemp < 21.0 ) ZoneNumber = 33;
		else if( LongTemp >= 21.0 && LongTemp < 33.0 ) ZoneNumber = 35;
		else if( LongTemp >= 33.0 && LongTemp < 42.0 ) ZoneNumber = 37;
    }

	LongOrigin = (ZoneNumber - 1)*6 - 180 + 3;  //+3 puts origin in middle of zone
	LongOriginRad = LongOrigin * deg2rad;

	//compute the UTM Zone from latitude and longitude
	sprintf(UTMZone, "%d%c", ZoneNumber, UTMLetterDesignator(Lat));

	eccPrimeSquared = (eccSquared)/(1-eccSquared);

	N = a/sqrt(1-eccSquared*sin(LatRad)*sin(LatRad));
	T = tan(LatRad)*tan(LatRad);
	C = eccPrimeSquared*cos(LatRad)*cos(LatRad);
	A = cos(LatRad)*(LongRad-LongOriginRad);

	M = a*((1 - eccSquared/4 - 3*eccSquared*eccSquared/64 - 5*eccSquared*eccSquared*eccSquared/256)*LatRad
			- (3*eccSquared/8+3*eccSquared*eccSquared/32	+ 45*eccSquared*eccSquared*eccSquared/1024)*sin(2*LatRad)
			+ (15*eccSquared*eccSquared/256 + 45*eccSquared*eccSquared*eccSquared/1024)*sin(4*LatRad)
			- (35*eccSquared*eccSquared*eccSquared/3072)*sin(6*LatRad));

	UTMEasting = (double)(k0*N*(A+(1-T+C)*A*A*A/6
					+ (5-18*T+T*T+72*C-58*eccPrimeSquared)*A*A*A*A*A/120)
					+ 500000.0);

	UTMNorthing = (double)(k0*(M+N*tan(LatRad)*(A*A/2+(5-T+9*C+4*C*C)*A*A*A*A/24
						+ (61-58*T+T*T+600*C-330*eccPrimeSquared)*A*A*A*A*A*A/720)));

	if(Lat < 0)
		UTMNorthing += 10000000.0; //10000000 meter offset for southern hemisphere
}

char GPSReceiver::UTMLetterDesignator(double Lat) {

	//This routine determines the correct UTM letter designator for the given latitude
	//returns 'Z' if latitude is outside the UTM limits of 84N to 80S
	//Written by Chuck Gantz- chuck.gantz@globalstar.com
	char LetterDesignator;

	if((84 >= Lat) && (Lat >= 72)) LetterDesignator = 'X';
	else if((72 > Lat) && (Lat >= 64)) LetterDesignator = 'W';
	else if((64 > Lat) && (Lat >= 56)) LetterDesignator = 'V';
	else if((56 > Lat) && (Lat >= 48)) LetterDesignator = 'U';
	else if((48 > Lat) && (Lat >= 40)) LetterDesignator = 'T';
	else if((40 > Lat) && (Lat >= 32)) LetterDesignator = 'S';
	else if((32 > Lat) && (Lat >= 24)) LetterDesignator = 'R';
	else if((24 > Lat) && (Lat >= 16)) LetterDesignator = 'Q';
	else if((16 > Lat) && (Lat >= 8)) LetterDesignator = 'P';
	else if(( 8 > Lat) && (Lat >= 0)) LetterDesignator = 'N';
	else if(( 0 > Lat) && (Lat >= -8)) LetterDesignator = 'M';
	else if((-8 > Lat) && (Lat >= -16)) LetterDesignator = 'L';
	else if((-16 > Lat) && (Lat >= -24)) LetterDesignator = 'K';
	else if((-24 > Lat) && (Lat >= -32)) LetterDesignator = 'J';
	else if((-32 > Lat) && (Lat >= -40)) LetterDesignator = 'H';
	else if((-40 > Lat) && (Lat >= -48)) LetterDesignator = 'G';
	else if((-48 > Lat) && (Lat >= -56)) LetterDesignator = 'F';
	else if((-56 > Lat) && (Lat >= -64)) LetterDesignator = 'E';
	else if((-64 > Lat) && (Lat >= -72)) LetterDesignator = 'D';
	else if((-72 > Lat) && (Lat >= -80)) LetterDesignator = 'C';
	else LetterDesignator = 'Z'; //This is here as an error flag to show that the Latitude is outside the UTM limits

	return LetterDesignator;
}

