//! @file ioctls.c
//! @brief IOCTLS examples for LEDS, digital IO and reset
//!
//! In this file LEDS, reset push button and digital IO can be tested
//! @author (C) 2006 VScom (www.vscom.de)

//! @def GOTO_ERROR(str)
//! @brief error output macro. Shows file name and line number
//! @param str error string

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include "vsopenrisc.h"

#define TEST_ERROR -1 //!< failure return value
#define GOTO_ERROR(str)	\
	do { \
		printf("%s(%d) / %s: %s (%d)\n", __FILE__, __LINE__, str, strerror(errno), errno); \
		goto error; \
	} while(0)

#define GPIO_BIT_0 0x01		//!< bit 0
#define GPIO_BIT_1 0x02		//!< bit 1
#define GPIO_BIT_2 0x04		//!< bit 2
#define GPIO_BIT_3 0x08		//!< bit 3
#define GPIO_BIT_4 0x10		//!< bit 4
#define GPIO_BIT_5 0x20		//!< bit 5
#define GPIO_BIT_6 0x40		//!< bit 6
#define GPIO_BIT_7 0x80		//!< bit 7

char *leds_txt [3] = {"POWER","BLUE","GREEN"};

//! @brief shows the content of the /proc/vsopenrisc/gpio_ctrl file
//! @param val content of the /proc/vsopenrisc/gpio_ctrl file that should be parsed
void vsopenrisc_show_gpio_ctrl(unsigned long val)
{
	unsigned char gpio_val = val;

	printf("GPIO CTRL:\n");
	printf("D0: %s\n",val & GPIO_BIT_0?"output":"input");
	printf("D1: %s\n",val & GPIO_BIT_1?"output":"input");
	printf("D2: %s\n",val & GPIO_BIT_2?"output":"input");
	printf("D3: %s\n",val & GPIO_BIT_3?"output":"input");
	printf("D4: %s\n",val & GPIO_BIT_4?"output":"input");
	printf("D5: %s\n",val & GPIO_BIT_5?"output":"input");
	printf("D6: %s\n",val & GPIO_BIT_6?"output":"input");
	printf("D7: %s\n\n",val & GPIO_BIT_7?"output":"input");
	printf("Value: 0x%02X\n\n", gpio_val);
}

//! @brief shows the content of the /proc/vsopenrisc/gpio_data file
//! @param val content of the /proc/vsopenrisc/gpio_data file that should be parsed
void vsopenrisc_show_gpio_data(unsigned long val)
{
	unsigned char gpio_val = val;

	printf("GPIO DATA:\n");
	printf("D0: %s\n",val & GPIO_BIT_0?"1":"0");
	printf("D1: %s\n",val & GPIO_BIT_1?"1":"0");
	printf("D2: %s\n",val & GPIO_BIT_2?"1":"0");
	printf("D3: %s\n",val & GPIO_BIT_3?"1":"0");
	printf("D4: %s\n",val & GPIO_BIT_4?"1":"0");
	printf("D5: %s\n",val & GPIO_BIT_5?"1":"0");
	printf("D6: %s\n",val & GPIO_BIT_6?"1":"0");
	printf("D7: %s\n\n",val & GPIO_BIT_7?"1":"0");
	printf("Value: 0x%02X\n\n", gpio_val);
}

//! @brief shows the content of the /proc/vsopenrisc/gpio_change file
//! @param val content of the /proc/vsopenrisc/gpio_change file that should be parsed
void vsopenrisc_show_gpio_change(unsigned long val)
{
	unsigned char gpio_val = val;

	printf("GPIO CHANGE\n");
	printf("D0: %s\n",val & GPIO_BIT_0?"1":"0");
	printf("D1: %s\n",val & GPIO_BIT_1?"1":"0");
	printf("D2: %s\n",val & GPIO_BIT_2?"1":"0");
	printf("D3: %s\n",val & GPIO_BIT_3?"1":"0");
	printf("D4: %s\n",val & GPIO_BIT_4?"1":"0");
	printf("D5: %s\n",val & GPIO_BIT_5?"1":"0");
	printf("D6: %s\n",val & GPIO_BIT_6?"1":"0");
	printf("D7: %s\n\n",val & GPIO_BIT_7?"1":"0");
	printf("Value: 0x%02X\n\n", gpio_val);
}

//! @brief shows the content of the /proc/vsopenrisc/gpio_irqmask file
//! @param val content of the /proc/vsopenrisc/gpio_irqmask file that should be parsed
void vsopenrisc_show_gpio_irqmask(unsigned long val)
{
	unsigned char gpio_val = val;

	printf("GPIO IRQMASK\n");
	printf("D0: %s\n",val & GPIO_BIT_0?"1":"0");
	printf("D1: %s\n",val & GPIO_BIT_1?"1":"0");
	printf("D2: %s\n",val & GPIO_BIT_2?"1":"0");
	printf("D3: %s\n",val & GPIO_BIT_3?"1":"0");
	printf("D4: %s\n",val & GPIO_BIT_4?"1":"0");
	printf("D5: %s\n",val & GPIO_BIT_5?"1":"0");
	printf("D6: %s\n",val & GPIO_BIT_6?"1":"0");
	printf("D7: %s\n\n",val & GPIO_BIT_7?"1":"0");
	printf("Value: 0x%02X\n\n", gpio_val);
}

//! @brief shows the content of the /proc/vsopenrisc/leds file
//! @param val content of the /proc/vsopenrisc/gpio_leds file that should be parsed
void vsopenrisc_show_leds(unsigned long val)
{
	printf("\nLEDS STATUS\n");

	if(val & LED_POWER)
		printf("LED POWER: on\n");
	else
		printf("LED POWER: off\n");

	if(val & LED_BLUE)
		printf("LED BLUE: on\n");
	else
		printf("LED BLUE: off\n");

	if(val & LED_GREEN)
		printf("LED GREEN: on\n");
	else
		printf("LED GREEN: off\n");

	printf("Value: 0x%02lX\n\n", val);
}

//! @brief main function
//! @param argc number of command line arguments
//! @param argv command line arguments
int main(int argc, char *argv[])
{
	int fd = -1, one = 1, zero = 0, led_idx, i;
	unsigned long val;

	fd = open("/dev/gpio", O_RDWR);
	if (fd < 0)
		GOTO_ERROR("open");

	////////////
	// BUZZER //
	////////////

	printf("BUZZER PART\n\n");

	if(ioctl(fd, GPIO_CMD_GET_BUZZER, &val) == TEST_ERROR)
		perror("ioctl: GPIO_CMD_GET_BUZZER");

	printf("Buzzer 0x%08lX\n", val);

	printf("Switching buzzer on\n");
	val = 1UL;
	if (ioctl(fd, GPIO_CMD_SET_BUZZER, &val) == -1)
		perror("ioctl: GPIO_CMD_SET_BUZZER");

	sleep(1);

	if(ioctl(fd, GPIO_CMD_GET_BUZZER, &val) == TEST_ERROR)
		perror("ioctl: GPIO_CMD_GET_BUZZER");

	printf("Buzzer 0x%08lX\n", val);

	printf("Switching buzzer off\n");
	val = 0UL;
	if (ioctl(fd, GPIO_CMD_SET_BUZZER, &val) == -1)
		perror("ioctl: GPIO_CMD_SET_BUZZER");

	sleep(1);

	if(ioctl(fd, GPIO_CMD_GET_BUZZER, &val) == TEST_ERROR)
		perror("ioctl: GPIO_CMD_GET_BUZZER");


	printf("Buzzer 0x%08lX\n\n", val);


	if(ioctl(fd, GPIO_CMD_GET_BUZZER_FRQ, &val) == TEST_ERROR)
		perror("ioctl: GPIO_CMD_GET_BUZZER_FRQ");


	printf("Buzzer frequency is set to 0x%08lXms\n", val);
	sleep(2);

	val = 0x000001f4UL;

	if (ioctl(fd, GPIO_CMD_SET_BUZZER_FRQ, &val) == -1)
		perror("ioctl: GPIO_CMD_SET_BUZZER_FRQ");

	if(ioctl(fd, GPIO_CMD_GET_BUZZER_FRQ, &val) == TEST_ERROR)
		perror("ioctl: GPIO_CMD_GET_BUZZER_FRQ");

	printf("Buzzer frequency is set to 0x%08lXms\n", val);
	sleep(5);

	val = 0x000007d0UL;
	if (ioctl(fd, GPIO_CMD_SET_BUZZER_FRQ, &val) == -1)
		perror("ioctl: GPIO_CMD_SET_BUZZER_FRQ");

	if(ioctl(fd, GPIO_CMD_GET_BUZZER_FRQ, &val) == TEST_ERROR)
		perror("ioctl: GPIO_CMD_GET_BUZZER_FRQ");


	printf("Buzzer frequency is set to 0x%08lXms\n", val);
	sleep(7);
	val = 0UL;
	if (ioctl(fd, GPIO_CMD_SET_BUZZER, &val) == -1)
		perror("ioctl: GPIO_CMD_SET_BUZZER");
	if(ioctl(fd, GPIO_CMD_GET_BUZZER_FRQ, &val) == TEST_ERROR)
		perror("ioctl: GPIO_CMD_GET_BUZZER_FRQ");

	printf("Switching buzzer off\n");
	printf("Buzzer frequency is set to 0x%08lXms\n", val);

	//////////
	// LEDS //
	//////////
	printf("LEDS PART\n");

	if(ioctl(fd, GPIO_CMD_GET_LEDS, &val) == TEST_ERROR)
		perror("ioctl: GPIO_CMD_GET_LEDS");

	vsopenrisc_show_leds(val);
	printf("blue led off\n");
	if (ioctl(fd, GPIO_CMD_SET_LED_BLUE, &zero) == -1)
		perror("ioctl: GPIO_CMD_SET_LED_BLUE");

	sleep(2);

	if(ioctl(fd, GPIO_CMD_GET_LEDS, &val) == TEST_ERROR)
		perror("ioctl: GPIO_CMD_GET_LEDS");

	vsopenrisc_show_leds(val);

	printf("blue led on\n");
	if (ioctl(fd, GPIO_CMD_SET_LED_BLUE, &one) == -1)
		perror("ioctl: GPIO_CMD_SET_LED_BLUE");

	sleep(2);

	if(ioctl(fd, GPIO_CMD_GET_LEDS, &val) == TEST_ERROR)
		perror("ioctl: GPIO_CMD_GET_LEDS");

	vsopenrisc_show_leds(val);

	printf("led switching\n\n");
	if (ioctl(fd, GPIO_CMD_SET_LEDS, &zero) == -1)
		perror("ioctl: GPIO_CMD_SET_LEDS");

	for (led_idx = 0, val = 1; val <= LED_GREEN; val <<= 1, led_idx++)
	{
		printf("0x%02lX LED %s\n", val, leds_txt[led_idx]);
		if (ioctl(fd, GPIO_CMD_SET_LEDS, &val) == -1)
			perror("ioctl: GPIO_CMD_SET_LEDS");
		sleep(1);
	}


	///////////
	// RESET //
	///////////
	printf("\nRESET PART\n");
	if (ioctl(fd, GPIO_CMD_GET_BTN_RST, &val) == -1)
		perror("ioctl: GPIO_CMD_GET_BTN_RST");

	if(val == 0)
		printf("Reset not pressed (0x%02lX)\n", val);
	else
		printf("Reset pressed (0x%02lX)\n", val);
	printf("\n");

	//////////////////////////////////
	// GPIOs (Irqs / Leds / Switch) //
	//////////////////////////////////
	printf("Digital IO PART\n\n");
	if(ioctl(fd, GPIO_CMD_GET, &val) == TEST_ERROR)
		perror("ioctl: GPIO_CMD_GET");

	vsopenrisc_show_gpio_data(val);

	if(ioctl(fd, GPIO_CMD_GET_CTRL, &val) == TEST_ERROR)
		perror("ioctl: GPIO_CMD_GET_CTRL");

	vsopenrisc_show_gpio_ctrl(val);
	if(ioctl(fd, GPIO_CMD_GET_CHANGE, &val) == TEST_ERROR)
		perror("ioctl: GPIO_CMD_GET_CHANGE");

	vsopenrisc_show_gpio_change(val);
	if(ioctl(fd, GPIO_CMD_GET_IRQMASK, &val) == TEST_ERROR)
		perror("ioctl: GPIO_CMD_GET_IRQMASK");

	vsopenrisc_show_gpio_irqmask(val);

	printf("GPIO CHANGES\n");
	for (i=0;i<NUMBER_OF_GPIOS;i++)
	{
		val = i;
		if(ioctl(fd, GPIO_CMD_GET_CHANGES, &val) == TEST_ERROR)
			perror("ioctl: GPIO_CMD_GET_CHANGES");

		printf("GPIO %d: %lu\n", i, val);
	}
	printf("\n");

	close(fd);

	return 0;
	error:
	if (fd >= 0)
		close(fd);
	return -1;
}

