//! @file ioctls2.c
//! @brief IOCTLS examples for UART and networking
//!
//! In this file UART and networking can be tested
//! @author Vision Systems GmbH (www.visionsystems.de)

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <arpa/inet.h>
#include <linux/if.h>
#include <string.h>
#include <sys/socket.h>
#include <linux/sockios.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include "vsopenrisc.h"


#define SER_DEV		"/dev/ttyS2"
#define NET_DEV		"eth0"

#define GOTO_ERROR(str)	\
	do { \
		printf("%s(%d) / %s: %s (%d)\n", __FILE__, __LINE__, str, strerror(errno), errno); \
		goto error; \
	} while(0)


static struct
{
	unsigned char value;
	char name[32];
} epld_str_tab[] =
{
		 { EPLD_RS232,          "rs232" }
		,{ EPLD_RS422,          "rs422" }
		,{ EPLD_RS485_ART_4W,   "rs485byart-4-wire" }
		,{ EPLD_RS485_ART_2W,   "rs485byart-2-wire-noecho" }
		,{ EPLD_RS485_ART_ECHO, "rs485byart-2-wire-echo" }
		,{ EPLD_RS485_RTS_4W,   "rs485byrts-4-wire" }
		,{ EPLD_RS485_RTS_2W,   "rs485byrts-2-wire-noecho" }
		,{ EPLD_RS485_RTS_ECHO, "rs485byrts-2-wire-echo" }
		,{ EPLD_PORTOFF,        "inactive" }
};
static int epld_str_tab_size = sizeof(epld_str_tab) / sizeof(epld_str_tab[0]);


int main(int argc, char *argv[])
{
	int i, fd = -1, s = -1;
	struct epld_struct epld;
	struct sockaddr_in *addrp;
	struct ifreq ifr;

	fd = open(SER_DEV, O_RDWR);
	if (fd < 0)
		GOTO_ERROR("open");


	//////////////////////////////////////////////////
	// read the current external serial port 2 mode //
	//////////////////////////////////////////////////
	printf("reading the mode of port %s\n", SER_DEV);

	if (ioctl(fd, TIOCGEPLD, &epld) < 0)
		perror("ioctl: TIOCGEPLD");

	for (i = 0; i < epld_str_tab_size; i++)
	{
		if (epld_str_tab[i].value == epld.value)
			printf("port %s is configured to %s\n", SER_DEV, epld_str_tab[i].name);
	}

	sleep(2);


	/////////////////////////////////////////////
	// set the external serial port 2 to rs422 //
	/////////////////////////////////////////////
	printf("setting %s to rs422 mode\n", SER_DEV);

	epld.value = EPLD_RS422;
	if (ioctl(fd, TIOCSEPLD, &epld) < 0)
		perror("ioctl: TIOCSEPLD");

	close(fd);

	sleep(2);


	/////////////////////////////////
	// read the current ip address //
	/////////////////////////////////
	strcpy(ifr.ifr_name, NET_DEV);
	addrp = (struct sockaddr_in*)&ifr.ifr_addr;
	addrp->sin_family = AF_INET;

	s = socket(AF_INET, SOCK_DGRAM, 0);
	if (s == -1)
		GOTO_ERROR("socket");

	if (ioctl(s, SIOCGIFADDR, &ifr) == -1)
		GOTO_ERROR("ioctl: SIOCGIFADDR");

	printf("the actual ip address is %s\n", inet_ntoa(addrp->sin_addr));

	sleep(2);


	///////////////////////////////////////
	// set the ip address to a new value //
	///////////////////////////////////////
	printf("set the ip address to a new value\n");

	addrp->sin_addr.s_addr = inet_addr("192.168.254.127");
	if (ioctl(s, SIOCSIFADDR, &ifr) == -1)
		GOTO_ERROR("ioctl: SIOCSIFADDR");

	close(s);

	return 0;
	error:
	if (s >= 0)
		close(s);
	if (fd >= 0)
		close(fd);
	return -1;
}

