//! @file rawsrv.c
//! @brief raw server example
//!
//! raw server awaits packets from network and sends them via serial interface and vice versa
//! @author Vision Systems GmbH (www.visionsystems.de)

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <termios.h>
#include <ctype.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include "vsopenrisc.h"


#define NET_PORT	3000
#define SER_DEV		"/dev/ttyS1"
#define BUF_SIZE	1024

#define GOTO_ERROR(str)	\
	do { \
		printf("%s(%d) / %s: %s (%d)\n", __FILE__, __LINE__, str, strerror(errno), errno); \
		goto error; \
	} while(0)


int mylisten(unsigned short port)
{
	int s = -1, s2 = -1, one = 1, size;
	struct sockaddr_in addr, addr2;

	s = socket(AF_INET, SOCK_STREAM, 0);
	if (s == -1)
		GOTO_ERROR("socket");

	if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, &one, sizeof(one)) == -1)
		GOTO_ERROR("setsockopt");

	memset(&addr, 0, sizeof(addr));

	addr.sin_addr.s_addr = INADDR_ANY;
	addr.sin_family = AF_INET;
	addr.sin_port = htons(port);

	if (bind(s, (struct sockaddr*)&addr, sizeof(addr)) == -1)
		GOTO_ERROR("bind");

	if (listen(s, 1) == -1)
		GOTO_ERROR("listen");

	printf("listening on tcp port %d\n", port);

	size = sizeof(addr2);
	s2 = accept(s, (struct sockaddr*)&addr2, &size);
	if (s2 == -1)
		GOTO_ERROR("accept");

	if (ioctl(s2, FIONBIO, &one) == -1)	// nonblocking
		GOTO_ERROR("ioctl");

	printf("connection from %s:%d\n", inet_ntoa(addr2.sin_addr), ntohs(addr2.sin_port));

	close(s);	// don't need the listen socket anymore

	return s2;
error:
	if (s >= 0)
		close(s);
	if (s2 >= 0)
		close(s2);
	return -1;
}

////////////////////////////////////////////////////////////////////////////////

int myseropen(char *dev)
{
	int fd = -1, one = 1;
	struct termios newtio;

	fd = open(dev, O_RDWR);
	if (fd == -1)
		GOTO_ERROR("open");

	if (tcgetattr(fd, &newtio) == -1)
		GOTO_ERROR("tcgetattr");

	cfmakeraw(&newtio);
	cfsetospeed(&newtio, B115200);
	cfsetispeed(&newtio, 0);	// same as ospeed

	if (tcsetattr(fd, TCSANOW, &newtio) == -1)
		GOTO_ERROR("tcsetattr");

	if (ioctl(fd, FIONBIO, &one) == -1)	// nonblocking
		GOTO_ERROR("ioctl");

	tcflush(fd, TCIOFLUSH);

	return fd;
error:
	if (fd >= 0)
		close(fd);
	return -1;
}

////////////////////////////////////////////////////////////////////////////////
// we read all the pending data until our buffer is full.

int recv_data(int fd, unsigned char *buf, int *psize)
{
	int rv, count, offset, size = *psize;

	while ((count = BUF_SIZE - size))
	{
		offset = BUF_SIZE - count;

		rv = read(fd, buf + offset, count);
		if (rv == -1)
		{
			if (errno == EAGAIN)
				goto exit;
			GOTO_ERROR("read");
		}
		if (rv)
		{
			size += rv;
			if (size == BUF_SIZE)
				break;
		}
		else
			return 0; // EOF (possible socket close / use getsockopt
					  //	  for more information -> SO_ERROR)
	}

exit:
	*psize = size;
	return count;
error:
	return -1;
}

////////////////////////////////////////////////////////////////////////////////
// we send as many data as we could and throw the rest away if any error occur.

int send_data(int fd, unsigned char *buf, int *psize)
{
	int rv, offset = 0, size = *psize;

	*psize = 0;

	while (size > 0)
	{
		rv = write(fd, buf + offset, size);
		if (rv == -1)
		{
			if (errno == EAGAIN)
				continue;
			GOTO_ERROR("read");
		}
		if (rv)
		{
			size -= rv;
			offset += rv;
		}
	}

	return 0;
error:
	return -1;
}

////////////////////////////////////////////////////////////////////////////////

int filter_data(unsigned char *buf, int *psize)
{
	int i, size = *psize;

	for (i = 0; i < size; i++)
	{
		if (isupper(buf[i]))
			buf[i] = tolower(buf[i]);
		else
			buf[i] = toupper(buf[i]);
	}

	return 0;
}

////////////////////////////////////////////////////////////////////////////////

void hex_dump(unsigned char *buf, int size, char add)
{
	static int count = 0;
	int i;

	for (i = 0; i < size; i++)
	{
		count++;
		printf("%c%02X(%c)", add, buf[i], isprint(buf[i]) ? buf[i] : '*');
		if (!(count % 8))
			printf("\n");
		else
		{
			if (!(count % 4))
				printf(" - ");
			else
				printf(" ");
		}
	}

	fflush(stdout);
}

////////////////////////////////////////////////////////////////////////////////

int main(int argc, char *argv[])
{
	int fd = -1, s = -1, rv, max_fd_val = 0;
	int size_sock = 0, size_ser = 0;
	unsigned char buf_sock[BUF_SIZE], buf_ser[BUF_SIZE];
	fd_set fds_in;

	s = mylisten(NET_PORT);
	if (s == -1)
		GOTO_ERROR("mylisten");

	fd = myseropen(SER_DEV);
	if (fd == -1)
		GOTO_ERROR("myseropen");

	max_fd_val = (s > fd) ? s : fd;

	for (;;)
	{
		FD_ZERO(&fds_in);
		FD_SET(s, &fds_in);
		FD_SET(fd, &fds_in);

		rv = select(max_fd_val + 1, &fds_in, NULL, NULL, NULL);
		if (rv == -1)
			GOTO_ERROR("select");
		else
		{
			if (FD_ISSET(s, &fds_in))
			{
				if (!recv_data(s, buf_sock, &size_sock))
				{
					printf("socket get closed?\n");
					goto error;
				}
			}
			if (FD_ISSET(fd, &fds_in))
			{
				if (!recv_data(fd, buf_ser, &size_ser))
				{
					printf("serial port get closed?\n");
					goto error;
				}
			}
			if (size_sock)
			{
				hex_dump(buf_sock, size_sock, '.');
				filter_data(buf_sock, &size_sock);
				send_data(fd, buf_sock, &size_sock);
			}
			if (size_ser)
			{
				hex_dump(buf_ser, size_ser, ' ');
				filter_data(buf_ser, &size_ser);
				send_data(s, buf_ser, &size_ser);
			}
			// both buffers are empty now ;) -> take care of send_data
		}
	}

	close(fd);
	close(s);

	return 0;
error:
	if (fd >= 0)
		close(fd);
	if (s >= 0)
		close(s);
	return -1;
}

