// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  DAConverter.cpp
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Stephan Riedel, stephan-riedel@gmx.de
// Date:      25.07.2008
// *****************************************************************

#include "iic/ADConverter.h"

using namespace std;

IICADConverter::IICADConverter(IICBus& bus, int addr, const string& name, eDAConverterMode mode)
	: IICBase(bus,addr,name), m_mode(mode), m_channel(-1), m_autoinc(0) {
}

void IICADConverter::init() {

	for (int i=0; i<4; i++) scoreTable[i].clear();

	m_control_byte = 0x00;
	m_control_byte |= (char)m_mode << 4;
	writeData(&m_control_byte,1);
}

int IICADConverter::setAutoInc(bool autoinc){

	m_autoinc = autoinc;
	if ( autoinc ) {
		m_control_byte |= 0x04; // set bit 2
		return writeData(&m_control_byte,1);
	}
	else {
		m_control_byte &= ~0x04; // clear bit 2
		return writeData(&m_control_byte,1);
	}
}

int IICADConverter::setOutput(int value) {

	if ( value == -1 ) {
		m_control_byte &= ~0x40; // clear bit 6
		return writeData(&m_control_byte,1);
	}
	else {
		m_control_byte |= 0x40; // set bit 6
		char buf[2];
		buf[0] = m_control_byte;
		buf[1] = value;
		return writeData(buf,2);
	}
}

int IICADConverter::setChannel(int channel){

	int max_channel = MAXCH;
	if (m_mode == THREE_DIFF_INPUTS) max_channel = 3;
	else if (m_mode == SINGLE_DIFF_MIXED) max_channel = 3;
	else if (m_mode == TWO_DIFF_INPUTS) max_channel = 2;

	if (channel >= max_channel || channel < 0) {
		printf("IICADConverter::setChannel(): channel %d not available in this mode!\n",channel);
		return -1;
	}

	m_control_byte |= channel; // set bit 6
	if (writeData(&m_control_byte,1) == 1) {
		m_channel = channel;
		return m_channel;
	}
	else return -1;
}

int IICADConverter::readChannel(int channel, bool read_immediate, int table_num) {

	if (!m_autoinc && channel != m_channel) { // only set new channel if auto-inc is off and new != old
		setChannel(channel);
	}
	if (table_num < -1 || table_num >= MAXCH) return -1;

	char buf[2];
	if (readData(buf,1 + read_immediate) == 1 + read_immediate) { // read last converted byte and/or new byte
		int val = buf[read_immediate] & 0x00FF;
		if (table_num != -1) return getScoreTable(val,table_num);
		else return val;
	}
	else return -1;
}

void IICADConverter::setScoreTable(int v1, int v2, int channel) {

	if (channel < 0 || channel >= MAXCH) return;
	scoreTable[channel].insert(pair<int,int>(v1,v2));
}

int IICADConverter::getScoreTable(int value, int channel) {

	if (channel < 0 || channel >= MAXCH || scoreTable[channel].size() == 0) return -1;

	map<int, int>::iterator it_upper = scoreTable[channel].upper_bound( value );
	map<int, int>::iterator it_lower = it_upper;
	if (it_upper == scoreTable[channel].end()) {
		it_upper--;
		it_lower--;
	}
	else if (it_upper != scoreTable[channel].begin()) it_lower--;

	if (it_upper->first - it_lower->first == 0) return it_lower->second;
	else { // linear interpolation
		return ((it_upper->second - it_lower->second) * (value - it_lower->first)) /
		(it_upper->first - it_lower->first) + it_lower->second;
	}
}


