#ifndef IIC_BASE_H_
#define IIC_BASE_H_

#include <iostream>

#define MAXERR 5

class IICBus; // forward declaration

//! Base Class Definition for all iic classes.
class IICBase
{

public:

	IICBase(IICBus& bus, int addr, const std::string& name);
	virtual ~IICBase();

	//! Abstract method thart sub-classes override to do their initialization.
	virtual void init() = 0;

	//! Returns the name of this component
	const std::string& getName() const { return m_name; }

	//! Returns error count.
	int getErrCount() const { return m_err_count; }

	//! Retunrs the I²C address of this component.
	int getAddress() const { return m_addr; }

	//! Returns the status of this component.
	bool getStatus() const { return m_status; }

	//! Sets the module status/activity to TRUE or FALSE. No read/write operations are executed in false state.
	void setStatus(bool status);

	//! Marks the module as removable (rem=1). This avoids error acccumulation and automatic deactivation in case of removed iic chipcards.
	void setIsRemovable(unsigned char rem) { m_is_removable = rem; }

	//! Sends raw data to this component.
	int writeData(char* buf, int num);
	//! Receive raw data to this component.
	int readData(char* buf, int num);

private:
	int m_addr;						//!< i2c-address
	int m_id;						//!< module id (for bus identification)
	std::string m_name;				//!< module name
	bool m_status;					//!< module status (true/false)
	unsigned char m_is_removable;	//!< indicates removable device (chipcard)
	int m_err_count;				//!< counts read/write failures
	IICBus& m_bus;					//!< reference to connected iicbus
};

#endif /*IIC_BASE_*/
