#include <sys/ioctl.h>
#include <fcntl.h>
#include <linux/i2c-dev.h>
#include <linux/i2c.h>

#include "iic/Bus.h"

using namespace std;


IICBus::IICBus(const string& name, int iface_type, const string& iow_serial) : m_name(name), m_iface_type(iface_type) {

	MutexLocker m(m_io_mutex);

	m_curr_channel = -1; // current channel is off per default
	mp_curr_mux = 0x00;	// set current multiplexer to null pointer

	// find correct iowarrior interface if using iow
	if (m_iface_type ==1) {	// iowarrior
		iow_print_warriors(name.c_str());

		m_fd = iow_find_special_descriptor(iow_serial.c_str(), name.c_str());

		if ( m_fd  < 0) {
			printf("IICBus::IICBus(): IOW Device with serial 0x%s could not be found\n", iow_serial.c_str());
			exit(0);
		}

		m_iow_repsize = iow_i2c_init(m_fd);

	}
	else { //or native iic
		if ((m_fd = open(m_name.c_str(), O_RDWR)) < 0) {
			cout << "IICBus::IICBus() : Error on open() " << m_name << endl;
			exit(0);
		}
	}
}

IICBus::~IICBus() {
	MutexLocker m(m_io_mutex);

	close(m_fd);
}


void IICBus::init() {

	map<int, modinfo_t>::iterator m_iter;

	checkConfig();

	cout << "Initializing " << m_modules.size() << " devices on bus " << getName() << " :" << endl;

	for( m_iter = m_modules.begin(); m_iter != m_modules.end(); m_iter++ ) {
		cout << m_iter->second.p_module->getName() << endl;
		m_iter->second.p_module->init();
	}
}

int IICBus::autodetect() {

	MutexLocker m(m_io_mutex);

	char val = 0x00;
	int addr;
	int res;
	int counter = 0;

	// find correct iowarrior interface if using iow
	if (m_iface_type ==1) {	// iowarrior

		for (addr=1; addr <= 127; addr++) {
			res = iow_i2c_read(addr, &val, 1, m_iow_repsize, m_fd);
			if (res == 1) {
				cout << "Found device with address " << hex << "0x" << addr << dec << endl;
				counter++;
			}
		}

	}
	else { //or native iic

		for (addr=1; addr <= 127; addr++) {

			if ( ioctl(m_fd, I2C_SLAVE_FORCE, addr) < 0 ) {
				cout << "IICBus::iic_autodetect() : Error in ioctl() for addr " << hex << "0x" << addr << dec << endl;
				return -1;
			}

			res = read(m_fd,&val,1);
			if (res == 1) {
				cout << "Found device with address " << hex << "0x" << addr << dec << endl;
				counter++;
			}
		}
	}

	return counter;
}

int IICBus::addModule(int addr, IICBase* module) {

	map<int, modinfo_t>::iterator m_iter;

	modinfo_t modinfo;

	modinfo.addr = addr;
	modinfo.p_module = module;
	modinfo.p_mux = NULL;
	modinfo.mux_channel = 0;

	int id = 0;
	for( m_iter = m_modules.begin(); m_iter != m_modules.end(); m_iter++ ) {
		if (m_iter->first != id) break;
		id++;
	}

	m_modules.insert( make_pair(id, modinfo) );
	return id;
}

void IICBus::rmModule(int id) {

	map<int, modinfo_t>::iterator m_iter = m_modules.find(id);
	m_modules.erase(m_iter);
}

void IICBus::makeSubModule(IICBase& module,IICMultiplexer& mux, int channel){

	map<int, modinfo_t>::iterator m_iter;

	for( m_iter = m_modules.begin(); m_iter != m_modules.end(); m_iter++ ) {
		if (&module == m_iter->second.p_module) {
			m_iter->second.p_mux = &mux;
			m_iter->second.mux_channel = channel;
			break;
		}
	}
}


void IICBus::printModules() {

	map<int, modinfo_t>::iterator m_iter;

	cout << "Device list for bus " << m_name << " :" << endl;

	for(m_iter = m_modules.begin(); m_iter != m_modules.end(); m_iter++) {
		string mux= "none";
		int ch = -1;
		if (m_iter->second.p_mux != NULL) {
			mux=m_iter->second.p_mux->getName();
			ch=m_iter->second.mux_channel;
		}
		cout << "ID: " << m_iter->first << " Address: " << hex << "0x" << m_iter->second.addr << dec <<  " Name: " << m_iter->second.p_module->getName() <<
		" Mux: " << mux << " Channel: " << ch << endl;
	}

}


void IICBus::checkConfig() {
	map<int, modinfo_t>::iterator iter_1;
	map<int, modinfo_t>::iterator iter_2;

	for(iter_1=m_modules.begin(); iter_1!=m_modules.end(); iter_1++) {
		for(iter_2=m_modules.begin(); iter_2!=m_modules.end(); iter_2++) {
			if(iter_1 == iter_2) continue;
			if (iter_1->second.p_mux == iter_2->second.p_mux && iter_1->second.mux_channel == iter_2->second.mux_channel && iter_1->second.addr == iter_2->second.addr) {
				cout << "Failure on devices: " << endl;
				cout << iter_1->second.p_module->getName() << " (ID: " << iter_1->first << " Address: " << hex << "0x" << iter_1->second.addr << dec <<
				" Mux: " << iter_1->second.p_mux->getName() << " Channel: " << iter_1->second.mux_channel << ")" << endl;
				cout << iter_2->second.p_module->getName() << " (ID: " << iter_2->first << " Address: " << hex << "0x" << iter_2->second.addr << dec <<
				" Mux: " << iter_2->second.p_mux->getName() << " Channel: " << iter_2->second.mux_channel << ")" << endl;
				exit (0);
			}
		}
	}
}

int IICBus::iicWrite(int id, char* buf, int num) {
	MutexLocker m(m_io_mutex);

	map<int, modinfo_t>::iterator m_iter;

	m_iter = m_modules.find(id);
	if (m_iter == m_modules.end()) {
		return -1;
	}

	// CHECK IF SUB_BUS NEEDS SWITCH
	if (m_iter->second.p_mux != NULL) {

		if (mp_curr_mux != m_iter->second.p_mux) { // new multiplexer?
			if (mp_curr_mux != NULL) mp_curr_mux->setChannel(-1); // disable multiplexer
			mp_curr_mux = m_iter->second.p_mux; // remember new multiplexer
		}
		if (m_curr_channel != m_iter->second.mux_channel) { // just switch channels if needed
			if (mp_curr_mux->setChannel(m_iter->second.mux_channel) == m_iter->second.mux_channel) {
				m_curr_channel = m_iter->second.mux_channel; // remember new channel
			}
		}
	}

	// which interface are we working with
	if (m_iface_type ==1) {	// iowarrior
		return iow_i2c_write(m_iter->second.addr, buf, num, m_iow_repsize, m_fd);
	}

	else { //or native iic
		if ( ioctl(m_fd, I2C_SLAVE, m_iter->second.addr) < 0 ) {
			cout << "IICBus::iic_write() : Error in ioctl() for addr " << hex << "0x" << m_iter->second.addr << dec << endl;
			return -1;
		}
		return write(m_fd, buf, num);
	}

}

int IICBus::iicRead(int id, char* buf, int num) {
	MutexLocker m(m_io_mutex);

	map<int, modinfo_t>::iterator m_iter;

	m_iter = m_modules.find(id);
	if (m_iter == m_modules.end()) return -1;

	// CHECK IF SUB_BUS NEEDS SWITCH
	if (m_iter->second.p_mux != NULL) {

		if (mp_curr_mux != m_iter->second.p_mux) {
			if (mp_curr_mux != NULL) mp_curr_mux->setChannel(-1);
			mp_curr_mux = m_iter->second.p_mux;
		}
		if (m_curr_channel != m_iter->second.mux_channel) {
			if (mp_curr_mux->setChannel(m_iter->second.mux_channel) == m_iter->second.mux_channel) {
				m_curr_channel = m_iter->second.mux_channel;
			}
		}
	}

	// which interface are we working with
	if (m_iface_type ==1) {	// iowarrior
		return iow_i2c_read(m_iter->second.addr, buf, num, m_iow_repsize, m_fd);
	}

	else { //or native iic
		if ( ioctl(m_fd, I2C_SLAVE, m_iter->second.addr) < 0 ) {
			cout << "IICBus::iic_read() : Error in ioctl() for addr " << hex << "0x" << m_iter->second.addr << dec << endl;
			return -1;
		}
		int ret = read(m_fd, buf, num);
		return ret;
	}
}

