#ifndef IIC_BUS_H_
#define IIC_BUS_H_

#include <iostream>
#include <fstream>
#include <map>

#include "iic/Base.h"
#include "iic/Multiplexer.h"
#include "tools/iowarrior_i2c.h"
#include "tools/Mutex.h"

class IICBase; // forward declaration

//! This class represents iic bus objects with iowarrior or native iic interface.
class IICBus
{

public:
	//! Struct holding all information for a module.
	struct modinfo_t {
		IICBase* p_module;		//!< pointer to base object
		int addr;				//!< i2c address
		IICMultiplexer* p_mux;	//!< pointer to multiplexer object
		int mux_channel;		//!< connected multiplexer channel
	};

	IICBus(const std::string& name, int iface_type, const std::string& iow_serial="");
	virtual ~IICBus();

	//! Returns the name of this IICBus.
	const std::string& getName() const { return m_name; }

	//! Initializes all modules connected to this bus by running their init() functions.
	void init();
	//! Tries to identify unknown iic modules on bus. Found addresses are printed to stdout. Returns number of found modules.
	int autodetect();
	//! Adds a module to this bus. Returns the ID for this module in the bus.
	int addModule(int addr, IICBase* module);
	//! Removes a module from the bus. id is the ID that was returned from addModule().
	void rmModule(int id);
	//! Marks a already added module as submodule, which is connected to channel "channel" of multiplexer "mux" and not to the main iic bus.
	void makeSubModule(IICBase& module, IICMultiplexer& mux, int channel);
	//! Print out all modules connected with this bus.
	void printModules();
	//! Raw send buf to module id.
	int iicWrite(int id, char* buf, int num);
	//! Raw receive data from module id.
	int iicRead(int id, char* buf, int num);

private:
	void checkConfig();

	std::string m_name;                 //!< bus name
	int m_fd;                           //!< file descr. for device (e.g./dev/i2c-0)
	int m_iface_type;                   //!< 0 = "native"; 1 = "iowarrior"
	int m_iow_repsize;                  //!< hold size of IOWarrior reports
	std::map<int,modinfo_t> m_modules;  //!< map to hold pairs id and device info
	IICMultiplexer* mp_curr_mux;        //!< active multiplexer at this time
	int m_curr_channel;                 //!< current channel for active mux
	Mutex m_io_mutex;                   //!< protect iic bus access
};

#endif /*IIC_BUS_*/


