// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  ChipCard.cpp
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder, Stephan Riedel
// Date:      26.04.2008
// *****************************************************************

// The chipcard is organised in pages with 64 byte in size.
// After writing 64 bytes of data, the internal filepointer rolls over to the
// first address of the current page. Writing more than 64 byte at once will
// will overwrite allready existing data. Thus the data ist split into 64 byte
// large block. (See m_WriteEEPROM())

#include <string>
#include "iic/ChipCard.h"

using namespace std;

IICChipCard::IICChipCard(IICBus& bus, int addr, const string& name) : IICBase(bus, addr, name) {

	if (sizeof(ChipcardHeader_t) > PAGESIZE ) {
		std::cout << "Header is larger than one page!" << std::endl;
		exit(0);
	}

	setIsRemovable(1);
}

void IICChipCard::setToAddress(unsigned int address) {

	// send only two bytes to set internal filepointer to "address"
	unsigned char buf[2];
	buf[0] = address >> 8;
	buf[1] = address;

	writeData((char*)buf, 2);
}

int IICChipCard::readSCHeader() {

	setToAddress(0);

	if (readData((char*)&m_Header, sizeof(ChipcardHeader_t)) == sizeof(ChipcardHeader_t)) {

		if (strcmp(m_Header.type_id, TYPEID) != 0) return -1;

		return 0;
	}
	else return -1;
}

void IICChipCard::printSCHeader() {

	if (readSCHeader() != -1) {
		std::cout << "Type-ID:          " << m_Header.type_id << std::endl;
		std::cout << "Card-Number:      " << m_Header.card_number << std::endl;
		std::cout << "Card-Description: " << m_Header.descr << std::endl;
		std::cout << endl;
	}
	else {
		std::cout << "IICChipCard::printSCHeader(): Read error or unknown type-id, please format header correctly" << std::endl;
	}
}

int IICChipCard::formatSCHeader(const char* card_number, const char* descr) {

	char buf[sizeof(ChipcardHeader_t)+2];
	strncpy(m_Header.type_id, TYPEID, sizeof(m_Header.type_id));
	strncpy(m_Header.card_number, card_number, sizeof(m_Header.card_number));
	strncpy(m_Header.descr, descr, sizeof(m_Header.descr));

	std::cout << "Formatting header with following information:" << std::endl;
	std::cout << "Type-ID:          " << m_Header.type_id << std::endl;
	std::cout << "Card-Number:      " << m_Header.card_number << std::endl;
	std::cout << "Card-Description: " << m_Header.descr << std::endl;

	buf[0]=0x00;  // set address to 0
	buf[1]=0x00;
	memcpy(&buf[2], &m_Header, sizeof(ChipcardHeader_t));  // copy header to buffer
	if (writeData(buf, 2+sizeof(ChipcardHeader_t)) < 0) return -1;  // write data

	if (readSCHeader() < 0) return -1;

	if(memcmp(&buf[2], &m_Header, sizeof(ChipcardHeader_t)) != 0 ) {
		std::cout << "IICChipCard::formatSCHeader(): Format Header failed!" << std::endl;
		return -1;
	}
	else {
		std::cout << "OK" << std::endl;
		return 0;
	}
}

int IICChipCard::readSCData(char* dest, unsigned long size) {

	// skip if type-ID is different
	if (readSCHeader() == -1) {
		std::cout << "IICChipCard::readSCData(): Unknown Type-ID, cannot read from card" << std::endl;
		return -1;
	}

	setToAddress(PAGESIZE);

	if (readData(dest, size) == size) {
		return size;
	}
	else return -1;
}

int IICChipCard::writeSCData(char* source, unsigned long size) {

	char buf[PAGESIZE + 2];
	// do not write any data if type-ID is different!
	if (readSCHeader() == -1) {
		std::cout << "IICChipCard::writeSCData(): Unknown Type-ID, cannot write to card" << std::endl;
		return -1;
	}

	if (size > CHIPCARDSIZE - PAGESIZE) {
		std::cout << "IICChipCard::writeSCData(): Data + Header is larger than chipcard size!" << std::endl;
		return -1;
	}

	int num_pages = ((size - 1) / PAGESIZE) + 1;  // Number of bytes to write

	for (int i = 0; i < num_pages; ++i) {

		int address = (i+1) * PAGESIZE;
		buf[0] = address >> 8;
		buf[1] = address;

		if ( i == num_pages - 1) {	// last page
			memcpy(&buf[2], source + PAGESIZE*i, size % PAGESIZE);
			if (writeData(buf, size % PAGESIZE + 2) < 0) return -1;
		}
		else { // complete pages
			memcpy(&buf[2], source + PAGESIZE*i, PAGESIZE);
			if (writeData(buf, PAGESIZE + 2) < 0) return -1;
		}
		usleep(10000);
	}
	return size;
}

bool IICChipCard::isAvailable() {

	char buf = 0x00;
	return (writeData(&buf, 1) == 1);
}



