// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  Display.cpp
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder
// Date:      26.04.2008
// *****************************************************************

#include "iic/Display.h"
#include <time.h>

using namespace std;

IICDisplay::IICDisplay(IICBus& bus, int addr, const std::string& name,
iic_disp_type dtype) : IICBase(bus,addr,name) {

	m_disptype = dtype;
	m_ts_freeze_end.tv_sec = 0;
	m_ts_freeze_end.tv_nsec = 0;
}


IICDisplay::~IICDisplay() {
	clear();
}

void IICDisplay::init() {

	m_disp_control = 0x0c;	// set DB4 (Disp on/off) and DB3 to enable by default
	m_backlight = 0x00;		// disable backlight by default

	// Init Routine
	m_buf = 0x30 >> (4 - LED_DATA_SHIFT);	// set display to 8 bit mode
	writeData(&m_buf,1);

	trigger_enable();
	usleep(5000);

	m_buf = 0x30 >> (4 - LED_DATA_SHIFT);	// set display to 8 bit mode
	writeData(&m_buf,1);
	trigger_enable();
	usleep(100);

	m_buf = 0x30 >> (4 - LED_DATA_SHIFT);	// set display to 8 bit mode
	writeData(&m_buf,1);
	trigger_enable();

	// Enable 4 bit mode
	m_buf = 0x20 >> (4 - LED_DATA_SHIFT);	// set display to 4 bit mode
	writeData(&m_buf,1);
	trigger_enable();
	usleep(100);

	// Function set
	m_buf = 0x20 >> (4 - LED_DATA_SHIFT);	// 	HB: select two-line-mode and 5x7 dot mode
	writeData(&m_buf,1);
	trigger_enable();
	m_buf = 0x08 <<  LED_DATA_SHIFT;	//	LB: select two-line-mode and 5x7 dot mode
	writeData(&m_buf,1);
	trigger_enable();

	// Entry mode set
	m_buf = 0x00 >> (4 - LED_DATA_SHIFT); // HB: Cursor right, no shift
	writeData(&m_buf,1);
	trigger_enable();
	m_buf = 0x06 << LED_DATA_SHIFT; // LB: Cursor right, no shift
	writeData(&m_buf,1);
	trigger_enable();

	// Display on/off
	setDisplay(false);
	clear();
}

void IICDisplay::putChar(char ch){

	if ( isFreeze()) return;

	m_buf = (ch & 0xF0) >> (4 - LED_DATA_SHIFT); // HB: four upper bits
	m_buf |= LED_RS; // set write
	m_buf |= m_backlight;
	writeData(&m_buf,1);
	trigger_enable();

	m_buf = (ch & 0x0F) << LED_DATA_SHIFT; // LB: four lower bits
	m_buf |= LED_RS; // set write
	m_buf |= m_backlight;
	writeData(&m_buf,1);
	trigger_enable();
}

void IICDisplay::setCursorPos(int row, int col) {

	// set DD RAM address (Display Data Ram = cursor position)
	char data;

	switch(row) {
		case 0:	data = 0x00;
						break;
		case 1:	data = 0x40;
						break;
		case 2:	if (m_disptype == SIZE_4x16) data = 0x10;
						if (m_disptype == SIZE_4x20) data = 0x14;
						break;
		case 3:	if (m_disptype == SIZE_4x16) data = 0x50;
						if (m_disptype == SIZE_4x20) data = 0x54;
						break;
	}
	data += col;

	data |= 0x80;	// set D6

	// send high byte
	m_buf = (data & 0xF0)  >> (4 - LED_DATA_SHIFT);	// use only four upper data bits
	m_buf |= m_backlight;
	writeData(&m_buf,1);
	trigger_enable();

	// send low byte
	m_buf = (data & 0x0F) <<  LED_DATA_SHIFT; 	// shift D0..D3 to D4..D7
	m_buf |= m_backlight;
	writeData(&m_buf,1);
	trigger_enable();
}

void IICDisplay::setCursorMode(bool curs, bool blink) {

	m_disp_control &= ~(!curs << 1); 	// delete
	m_disp_control |=  (curs << 1); 	// ...or set cursor

	m_disp_control &= ~(!blink); 	// delete
	m_disp_control |=  blink; 	// ...or set blink mode

	// first send a high byte dummy
	m_buf = m_backlight | 0x00;
	writeData(&m_buf,1);
	trigger_enable();

	// now send correct low byte
	m_buf = (m_disp_control & 0x0F) << LED_DATA_SHIFT;
	m_buf |= m_backlight;
	writeData(&m_buf,1);
	trigger_enable();
}

void IICDisplay::setDisplay(bool disp) {

	if ( isFreeze()) return;

	m_disp_control &= ~(!disp << 2); 	// turn off
	m_disp_control |=  (disp << 2); 	// ...or on

	// first send a high byte dummy
	m_buf = m_backlight | 0x00;
	writeData(&m_buf,1);
	trigger_enable();

	// now send correct low byte
	m_buf = (m_disp_control & 0x0F) << LED_DATA_SHIFT;
	m_buf |= m_backlight;
	writeData(&m_buf,1);
	trigger_enable();
}

void IICDisplay::trigger_enable(){

	m_buf |= (char)LCD_ENABLE;	// set enable high
	writeData(&m_buf,1);
	usleep(10);

	m_buf &= ~(char)LCD_ENABLE;	// set enable low
	writeData(&m_buf,1);
	usleep(10);
}


void IICDisplay::clear() {

	if ( isFreeze()) return;

	// Clear display
	m_buf = m_backlight | 0x00; // HB: Clear disp
	writeData(&m_buf,1);
	trigger_enable();

	m_buf = 0x01 << LED_DATA_SHIFT; // LB: Clear disp
	m_buf |= m_backlight;
	writeData(&m_buf,1);
	trigger_enable();
}

void IICDisplay::printBuf() {

	// print buffer until end of line occurs
	for(int i = 0; i < DISPBUFLENGTH;i++) {

		if (m_disp_buf[i] == '\n') break;
		else putChar(m_disp_buf[i]);
	}
}

void IICDisplay::setBacklight(bool bl) {

	if (bl) {
		m_backlight = (char)LCD_BACKLIGHT;
		m_buf |= (char)LCD_BACKLIGHT; 	// ...or on
	}
	else {
		m_backlight = 0x00;
		m_buf &= ~(char)LCD_BACKLIGHT;
	}

	// send changes
	writeData(&m_buf,1); // no trigger here!
}

void IICDisplay::freeze(unsigned int time_ms) {

		clock_gettime(CLOCK_MONOTONIC, &m_ts_freeze_end);
		m_ts_freeze_end.tv_sec += time_ms / 1000;
		m_ts_freeze_end.tv_nsec += time_ms * 1000000;
}

bool IICDisplay::isFreeze() {

	if ( m_ts_freeze_end.tv_sec == 0 && m_ts_freeze_end.tv_nsec == 0 ) return false;
	else {
		struct timespec now;
		clock_gettime(CLOCK_MONOTONIC, &now);
		if (now.tv_sec > m_ts_freeze_end.tv_sec ||
			(now.tv_sec == m_ts_freeze_end.tv_sec && now.tv_nsec > m_ts_freeze_end.tv_nsec)) {
			m_ts_freeze_end.tv_sec = 0;
			m_ts_freeze_end.tv_nsec = 0;
			return false;
		}
		else return true;
	}
}






