// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  Display.h
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder
// Date:      26.04.2008
// *****************************************************************

#ifndef _DISPLAY_H_
#define _DISPLAY_H_

#include <iostream>

#include "iic/Base.h"

#define DISPBUFLENGTH 20

// define connection PCF8574 <-> LCD here
#define LCD_BACKLIGHT       0x08
#define LCD_ENABLE          0x04
#define LED_RW              0x02
#define LED_RS              0x01
#define LED_DATA_SHIFT      4

//! Enumeration of possible display sizes.
enum iic_disp_type {
	SIZE_2x16,
	SIZE_4x16,
	SIZE_2x20,
	SIZE_4x20 };

//! This class represents lcd displays, that are driven by a PCF8574 iic port expander.
class IICDisplay : public IICBase
{

public:

	IICDisplay(IICBus& bus, int addr, const std::string& name, iic_disp_type dtype);
	virtual ~IICDisplay();

	//! Clears the display.
	void clear();
	//! Writes a char at current cursor position.
	void putChar(char ch);
	//! Sets the cursor to position row, col (starting at 0,0).
	void setCursorPos(int row, int col);
	//! Sets the cursor on (curs=true) or off (curs=false), and sets blink mode on (blink=true) or off (blink=false).
	void setCursorMode(bool curs, bool blink);
	//! Sets the display on (disp=true) or off (disp=false).
	void setDisplay(bool disp);
	//! Enables or disabled LED-backlight.
	void setBacklight(bool bl);
	//! Returns a pointer to the display buffer. Number of chars written to this address must not exceed DISPBUFLENGTH.
	char* getBuf() { return m_disp_buf; }
	//! Puts chars from buffer to display until the next newline (\n) is found.
	void printBuf();
	//! Freezes display for the next time_ms milliseconds and ignores clear(), putChar() and setDisplay() during this time.
	void freeze(unsigned int time_ms);

protected:
	void init();

private:
	void trigger_enable();
	bool isFreeze();

	iic_disp_type m_disptype;         //!< display type (dimensions)
	char m_backlight;                 //!< backlight status
	char m_disp_control;              //!< display status
	char m_disp_buf[DISPBUFLENGTH];   //!< display character buffer
	char m_buf;                       //!< i2c-rw-buffer
	struct timespec m_ts_freeze_end;  //!< end of freeze time in ms (no clear(), putChar() and setDisplay() operations)
};

#endif /* _DISPLAY_H_ */

