// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  Stepper.cpp
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder
// Date:      26.04.2008
// *****************************************************************

#include "iic/Stepper.h"
#include <time.h>

using namespace std;

//#define DEBUG

IICStepper::IICStepper(IICBus& bus, int addr, const string& name) : IICBase(bus,addr, name) {
}

/**************************************************************
Init function
**************************************************************/
void IICStepper::init() {

	hardStop();
	getFullStatus1();
	getFullStatus2();
	resetPosition();
}

int IICStepper::getFullStatus1(){

	buf[0] = 0x81;		// command byte
	if (writeData(buf,1) != 1) return -1;
	if (readData(buf,8) != 8) return -1;	// read 8 data bytes

	params.i_run			= (buf[1] >> 4) & ~0xF0;
	params.i_hold			= buf[1] & 0x0F;
	params.v_max			= (buf[2] >> 4) & ~0xF0;
	params.v_min			= buf[2] & 0x0F;
	params.shaft			= (buf[3] & 0x10) >> 4;
	params.step_mode		= ((buf[3] & 0x60) >> 5) & ~0xFC;
	params.acc				= buf[3] & 0x0F;
	params.acc_shape		= buf[3] & 0x80;

	status.motion				= (buf[5] >> 5) & ~0xF8;
	status.over_current_a 	= buf[5] & 0x08;
	status.over_current_b 	= buf[5] & 0x04;
	status.vdd_reset			= buf[4] & 0x80;
	status.cp_fail				= buf[5] & 0x01;
	status.ext_switch			= (buf[5] >> 4) & 0x1;
	status.step_loss			= buf[4] & 0x40;
	status.e_def				= buf[4] & 0x20;
	status.uv2					= buf[4] & 0x10;
	status.temp_info			= buf[4] & 0x03;
	status.temp_warn			= buf[4] & 0x04;
	status.temp_down			= buf[4] & 0x08;

#ifdef DEBUG
	for (int i = 0; i < 8; i++)
		cout << " byte " << i << " is 0x" << hex << (int)buf[i] << dec << endl;
#endif
	return 0;
}

int IICStepper::getFullStatus2(){

	buf[0] = 0xFC;		// command byte
	if (writeData(buf,1) != 1) return -1;
	if (readData(buf,8) != 8) return -1;	// read 8 data bytes

	actual_pos = (buf[1] << 8) | (buf[2] & ~0xFF00);
	target_pos = (buf[3] << 8) | (buf[4] & ~0xFF00);
	secure_pos = (buf[6] & 0x07) << 8 | (buf[5] & ~0xFF00);

#ifdef DEBUG
	for (int i = 0; i < 8; i++)
		cout << " byte " << i << " is 0x" << hex << (int)buf[i] << dec << endl;
#endif
	return 0;
}

int IICStepper::gotoSecurePosition() {

	buf[0] = 0x84;	// command byte
	return writeData(buf,1);
}

int IICStepper::hardStop() {

	buf[0] = 0x85;	// command byte
	return writeData(buf,1);
}

int IICStepper::resetPosition() {

	buf[0] = 0x86;	// command byte
	return writeData(buf,1);
}

int IICStepper::resetToDefault() {

	buf[0] = 0x87;	// command byte
	return writeData(buf,1);
}

int IICStepper::runInit(unsigned char v_max, unsigned char v_min, short pos1, short pos2) {

	buf[0] = 0x88;		// command
	buf[1] = 0xFF;
	buf[2] = 0xFF;
	buf[3] = v_max << 8 | v_min;
	buf[4] = pos1 >> 8;
	buf[5] = pos1;
	buf[6] = pos2 >> 8;
	buf[7] = pos2;

	return writeData(buf,8);
}


int IICStepper::setMotorParam( motor_params params, short secure_pos ) {

	buf[0] = 0x89;		// command
	buf[1] = 0xFF;
	buf[2] = 0xFF;
	buf[3] = params.i_run << 4 | params.i_hold;
	buf[4] = params.v_max << 4 | params.v_min;
	buf[5] = (( secure_pos >> 3) & 0xE0) | params.shaft << 4 | params.acc;
	buf[6] = secure_pos;
	buf[7] = params.acc_shape << 4 | params.step_mode << 2;

	return writeData(buf,8);
}


int IICStepper::setPosition(short pos) {

	buf[0] = 0x8B;		// command
	buf[1] = 0xFF;
	buf[2] = 0xFF;
	buf[3] = pos >> 8;
	buf[4] = pos;

	return writeData(buf,5);
}

int IICStepper::softStop() {

	buf[0] = 0x8F;		// command
	return writeData(buf,1);
}

int IICStepper::referenceInit(motor_params params, unsigned char v_max, unsigned char v_min, short ref_dist, short null_dist) {

	motor_params temp_params = params;
	temp_params.v_max = v_max;
	temp_params.v_min = v_min;

	// set temp params, disable secure position
	setMotorParam(temp_params, -1024);

	resetPosition(); // want to go ref_dist from actual position, so reset before
	setPosition(ref_dist);

	getFullStatus1();

	while( !getExtSwitch() ) {
		getFullStatus1();
		if ( !getMotion() ) {
			cout << "IICStepper::referenceInit(): Could not reach external switch within " << ref_dist <<
			" steps, performing HARD STOP for \"" << getName() << "\"" << endl;
			init();
			return -1;
		}
	}

	// reference switch is pressed now
	init();

	usleep(200000);

	setPosition(null_dist);
	getFullStatus1();

	while( getMotion() ) {
		getFullStatus1();
	}

	// reset params, disable secure position
	setMotorParam(params, -1024);
	resetPosition();
	return 0;
}

int IICStepper::waitForStop(unsigned int maxtime_ms) {

	struct timespec ts_start, ts_end;
	clock_gettime(CLOCK_MONOTONIC, &ts_start);

	getFullStatus1();
	while( getMotion() ) {
		getFullStatus1();
		clock_gettime(CLOCK_MONOTONIC, &ts_end);
		unsigned int diff_ms = (ts_end.tv_sec - ts_start.tv_sec) * 1000 + (ts_end.tv_nsec - ts_start.tv_nsec)/1000000;
		if ( diff_ms > maxtime_ms) {
			cout << "IICStepper::waitForStop(): Time of " << maxtime_ms <<
			" ms exceeded, performing HARD STOP for \"" << getName() << "\"" << endl;
			hardStop();
			return -1;
		}
	}
	return 0;
}



void IICStepper::printParam() {

	if (getStatus()) {
		getFullStatus1();

		cout << "--------------" << endl;
		cout << "Parameters of " << getName() << endl;
		cout << "Irun " << (int)params.i_run << endl;
		cout << "Ihold " << (int)params.i_hold << endl;
		cout << "Vmax " << (int)params.v_max << endl;
		cout << "Vmin " << (int)params.v_min << endl;
		cout << "Shaft " << (int)params.shaft << endl;
		cout << "StepMode " << (int)params.step_mode << endl;
		cout << "Acc " << (int)params.acc << endl;
		cout << "AccSshape " << (int)params.acc_shape << endl;

	}
	else {
		printf("IICStepper::printParam(): Stepper with address 0x%0x is OFF\n",getAddress());
	}

}

void IICStepper::printStatus() {

	if (getStatus()) {
		getFullStatus1();

		cout << "Status of " << getName() << endl;
		cout << "--------------" << endl;
		cout << "Motion " << (int)status.motion << endl;
		cout << "OverCurrA " << (int)status.over_current_a << endl;
		cout << "OverCurrB " << (int)status.over_current_b << endl;
		cout << "VDD Reset " << (int)status.vdd_reset << endl;
		cout << "CP Stat " << (int)status.cp_fail << endl;
		cout << "Ext Switch " << (int)status.ext_switch << endl;
		cout << "Step Loss " << (int)status.step_loss << endl;
		cout << "Elec. Defect " << (int)status.e_def << endl;
		cout << "Undervoltage " << (int)status.uv2 << endl;
		cout << "Temp Info " << (int)status.temp_info << endl;
		cout << "Temp Warning " << (int)status.temp_warn << endl;
		cout << "Temp Shutdown " << (int)status.temp_down << endl;

	}
	else {
		printf("IICStepper::printStatus(): Stepper with address 0x%0x is OFF\n",getAddress());
	}
}




