// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  Stepper.h
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder
// Date:      26.04.2008
// *****************************************************************

#ifndef _STEPPER_H_
#define _STEPPER_H_

#include <iostream>
#include <map>

#include "iic/Base.h"

//! Parameter structure for stepper motor.
struct motor_params {
	uint8_t i_run, i_hold;  //!< coil peak and hold current
	uint8_t v_max, v_min;   //!< max and min velocities
	uint8_t shaft;          //!< direction of motion (0=
	uint8_t step_mode;      //!< stepping mode
	uint8_t acc;            //!< acceleration
	uint8_t acc_shape;      //!< acceleration shape
};

//! Status of a stepper motor.
struct motor_status {
	uint8_t motion;         //!< status of motion
	uint8_t over_current_a; //!< over current coil A
	uint8_t over_current_b; //!< over current coil B
	uint8_t vdd_reset;      //!< digital supply reset
	uint8_t cp_fail;        //!< charge pump status
	uint8_t ext_switch;     //!< external switch status
	uint8_t step_loss;      //!< step loss
	uint8_t e_def;          //!< electrical defect
	uint8_t uv2;            //!< under voltage
	uint8_t temp_info;      //!< temperature information
	uint8_t temp_warn;      //!< temperature warning
	uint8_t temp_down;      //!< temperature shutdown
};


//! This class represents iic stepper motion control ICs of type TMC222.
class IICStepper : public IICBase
{

public:
	IICStepper(IICBus& bus, int addr, const std::string& name);
	virtual ~IICStepper(){};

	// Following methods implemented according to application
	// commands overview in TMC222 datasheet
	//! updates motor status and motor parameters with the values stored in the TMC222 chip.
	int getFullStatus1();
	//! Update target pos, actual pos and secure pos witht he values stored in the TMC222 chip.
	int getFullStatus2();
	//void getOTPParam(); not implemented here, see setOPTParams.c
	//! Move position to secure position.
	int gotoSecurePosition();
	//! Sends hard stop command.
	int hardStop();
	//! Sets both current and goal position to 0.
	int resetPosition();
	//! Overrides RAM content with OTP values.
	int resetToDefault();
	//! Starts initializing run to reference position. Careful, external switch is not checked. Use referenceInit() instead.
	int runInit(unsigned char v_max, unsigned char v_min, short pos1, short pos2);
	//! Sends new motor parameters to the chip.
	int setMotorParam( motor_params params, short secure_pos );
	//void setOTPParam(); not implemented here, see setOPTParams.c
	//! Sets new target position.
	int setPosition(short pos);
	//! Sends a soft stop command.
	int softStop();

	// More methods for convenient user interface
	//! Calls getFullStatus1() and returns new motor status.
	motor_status  getMotorStatus(){ getFullStatus1(); return status; }
	//! Calls getFullStatus1() and returns new motor parameters.
	motor_params  getMotorParams(){ getFullStatus1(); return params; }

	// Access methods for structs, data must be requested from TMC before
	//! Get temperature info. You need to call getFullStatus1() before this call to get the newest data.
	unsigned char getTempInfo() const { return status.temp_info; }
	//! Get switch value (on or off). You need to call getFullStatus1() before this call to get the newest data.
	bool getExtSwitch() const { return status.ext_switch; }
	//! Get motion flag. You need to call getFullStatus1() before this call to get the newest data.
	bool getMotion() const { return status.motion; }
	//! Get current position of stepper motor. You need to call getFullStatus2() before this call to get the newest data.
	short getActualPos() const { return actual_pos; }
	//! Get target position of stepper motor. You need to call getFullStatus2() before this call to get the newest data.
	short getTargetPos() const { return target_pos; }
	//! Get secure position of stepper motor. You need to call getFullStatus2() before this call to get the newest data.
	short getSecurePos() const { return secure_pos; }

	//! Starts initializing run to reference position and stops at external switch. Maximum distance and direction is given by ref_dist. After contecting the switch, motor does null_dist steps to final position. Reference drive is using speed values v_min and v_max and switching motor params to params after reference init.
	int referenceInit(motor_params params, unsigned char v_max, unsigned char v_min, short ref_dist, short null_dist);
	//! Waits maxtime_ms for the motor to stop. If time is exceeded and the motor does not stop, a hard stop command is executed (return value -1 otherwise 0).
	int waitForStop(unsigned int maxtime_ms);

	//! Calls getFullStatus1() and prints current motor parameter to stdout.
	void printParam();
	//! Calls getFullStatus1() and prints current motor status to stdout.
	void printStatus();

protected:
	void init();

private:
	motor_params params;						//!< motor parameters
	motor_status status;						//!< motor status
	short actual_pos, target_pos, secure_pos;	//!< 16-bit-positions
	char buf[10];								//!< rw-buffer
};

#endif /* _STEPPER_H_ */


