// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  TempSensor.cpp
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder
// Date:      26.04.2008
// *****************************************************************

#include "iic/TempSensor.h"

using namespace std;

IICTempSensor::IICTempSensor(IICBus& bus, int addr, const string& name, char res, bool t_out_pol) : IICBase(bus,addr, name), m_res(res), m_pol(t_out_pol) {

}

void IICTempSensor::init() {

	m_conf_reg = 0x00; // standard is continuous conversion
	m_conf_reg |= m_res << 2;
	m_conf_reg |= m_pol << 1;

	// write configuration
	m_buf[0] = 0xAC;
	m_buf[1] = m_conf_reg;
	writeData(m_buf,2);
}

int IICTempSensor::startConvert() {

	m_buf[0] = 0x51;
	return writeData(m_buf,1);
}

int IICTempSensor::stopConvert() {

	m_buf[0] = 0x22;
	return writeData(m_buf,1);
}

int IICTempSensor::setThermostat(int t_low, int t_high) {

	int temp = t_high * 25;
	m_buf[0] = 0xA1;		// write high thermostat temp
	m_buf[1] = temp >> 8;
	m_buf[2] = temp;
	if (writeData(m_buf,3) < 0) return -1;

	temp = t_low * 25;
	m_buf[0] = 0xA2;		// write low thermostat temp
	m_buf[1] = temp >> 8;
	m_buf[2] = temp;
	return writeData(m_buf,3);
}


int IICTempSensor::resetTempFlags() {

	m_conf_reg &= ~0x60;
	m_buf[0] = 0xAC;
	m_buf[1] = m_conf_reg;
	return writeData(m_buf,2);
}

int IICTempSensor::readTemp() {

	m_buf[0] = 0xAA;
	if (writeData(m_buf,1) < 0) return -999;
	if (readData(m_buf,2) < 0) return -999;
	short th = m_buf[0] << 8;
	short tl = m_buf[1] & 0x00FF;
	return (th | tl) / 25; // divide with 25, 30000 equals 1250 degs

}

bool IICTempSensor::getTempFlag(bool highlow) {

	m_buf[0] = 0xAC;
	writeData(m_buf,1);
	readData(m_buf,1);

	return (m_buf[0] &= (0x20 << highlow));
}


