/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <sys/time.h>
#include <linux/types.h>
#include "iowarrior.h"


#define USB_VENDOR_ID_CODEMERCS	1984
/* low speed iowarrior */
#define USB_DEVICE_ID_CODEMERCS_IOW40	0x1500
#define USB_DEVICE_ID_CODEMERCS_IOW24	0x1501
#define USB_DEVICE_ID_CODEMERCS_IOWPV1	0x1511
#define USB_DEVICE_ID_CODEMERCS_IOWPV2	0x1512
/* full speed iowarrior */
#define USB_DEVICE_ID_CODEMERCS_IOW56	0x1503

int main (int argc, char **argv) {
    /* for device information */
    struct iowarrior_info info;
    /* A buffer for reading the data, suitable for all IOWarrior-Products */
    unsigned char buffer[7];
    /* The file-descriptor of our device */
    int fd = -1;
    /* A result for the operations */
    int result=-1;	//result of read and select operations
    /* the write-set for select */
    fd_set wrfds;	
    /* the timeout for select */
    struct timeval tv;	
    int i=0;
    char *argConvError;	/* for parsing the arguments on the commandline */
    
    /* we try to open the device attached to /dev/usb/iowarrior0 */
    if(( fd = open( "/dev/usb/iowarrior0", O_RDWR)) < 0 ) {
    	perror( "iowarrior open failed" );
	exit( 1 );
    }
    /* lets see what product we have */
    if( ioctl( fd, IOW_GETINFO, &info)) {
	perror( "Unable to retrieve device info" );
	goto exit;
    }
    /* print the device information */
    printf("The device attached to /dev/usb/iowarrior0\n");
    printf("VendorId=%04x\n",info.vendor);
    printf("ProductId=%04x ",info.product);
    if(info.product==USB_DEVICE_ID_CODEMERCS_IOW40)
	printf("(IOWarrior40)\n");
    else if(info.product==USB_DEVICE_ID_CODEMERCS_IOW24)
	printf("(IOWarrior24)\n");
    else if(info.product==USB_DEVICE_ID_CODEMERCS_IOW56)
	printf("(IOWarrior56)\n");
    else
	printf("(Ooops, unknown device!)\n");
    printf("Serial=0x%s\nRevision=0x%04x\n",info.serial, info.revision); 
    printf("Speed=");
    if(info.speed==1)
	printf("Low Speed (USB 1.1)\n");
    else if(info.speed==2)
	printf("Full Speed (USB 1.1)\n");
    else if(info.speed==3)
	printf("High Speed (USB 2.0)\n");
    else
	printf("Ooops, speed is unknown!\n");
    printf("Power=%dmA\nPacketSize=%d\n",info.power, info.report_size); 
    /* checking for the IO-PIN interface */
    if(info.if_num!=0) {
	printf("Opps, this is not the IO-PIN interface\n");
	printf("Can't go on\n");
	goto exit;
    }    
    //get the value to be written from the commandline
    memset(&buffer,0xFF,sizeof(buffer));
    if(argc!=info.report_size+1) {
	printf("You must specify %d port-values (in hex-notation) on the commandline\n",info.report_size);
	goto exit;
    }
    for(i=0;i<info.report_size;i++) {
    	errno=0;
	buffer[i]=(unsigned char)strtoul(argv[i+1],&argConvError,16);
	if(errno || *argConvError!='\0') {
	    printf("error parsing argument %d\n",i);
	    goto exit;
	}
    }
    tv.tv_sec=10;	//we use a timeout of 10 seconds  
    tv.tv_usec=0; 
    FD_ZERO(&wrfds);	//clear the read-descriptors for select 
    FD_SET(fd, &wrfds);	 
    result=select(fd+1,NULL,&wrfds,NULL,&tv); 
    if(result==-1) { 
 	//error in select 
 	perror("Error in select"); 
    } 
    else if(result==0) { 
 	//select returned 0, no new data to be read 
 	printf("Timeout expired without any data written\n"); 
    } 
    else { 
 	if(FD_ISSET(fd,&wrfds)) { 
 	    //We got here, so we there should be able to write the data 
	    result=write(fd,buffer,info.report_size);
 	    if(result!=info.report_size) { 
 		//well thats not what we expected... 
 		int errcode=errno; 
 		if(errcode==ENODEV)  
 		    printf("Device was unplugged\n"); 
 		else 
 		    perror("Error in writing!"); 
 	    } 
 	    else { 
		printf("Write OK\n"); 
 	    } 
 	} 
    } 
 exit:
    close( fd );
    exit( 0 );
}
