// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  AES.cpp
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Stephan Riedel, stephan-riedel@gmx.de
// Date:      30.05.2008
// *****************************************************************

#include "AES.h"

using namespace std;

#define Nb 4  // The number of columns comprising a state in AES. In AES this is a constant value (=4).
#define xtime(x)   ((x<<1) ^ (((x>>7) & 1) * 0x1b))  // this is a macro that finds the product of {02} and the argument to xtime modulo {1b} - see MixColums()
#define Multiply(x,y) (((y & 1) * x) ^ ((y>>1 & 1) * xtime(x)) ^ ((y>>2 & 1) * xtime(xtime(x))) ^ ((y>>3 & 1) * xtime(xtime(xtime(x)))) ^ ((y>>4 & 1) * xtime(xtime(xtime(xtime(x))))))  // this is a macro which multiplies numbers in the field GF(2^8)


AES::AES() : m_is_encr(-1), m_data_decr_size(0), m_data_encr_size(0), mp_data_encr(NULL), mp_data_decr(NULL){

	const unsigned int bit_key_size = KEYSIZE * 8;	// key size in bits

	m_nk = bit_key_size / 32;
	m_nr = m_nk + 6;
}

AES::~AES() {
	free(mp_data_encr);
	free(mp_data_decr);
}

int AES::encrypt() {

	if (m_is_encr == -1) {
		cout << "No data imported yet" << endl;
		return -1;
	}
	else if (m_is_encr == 1) return -1;
	else {

		// The KeyExpansion routine must be called before encryption.
		keyExpansion();

		// encrypts data
		for (int i=0; i<m_nBlocks; ++i) {
			cipher(i);
		}
		m_is_encr = 1;
	}
	AESHeader header;
	strncpy(header.uin, UIN, UINSIZE);
	header.decr_size = m_data_decr_size;
	header.encr_size = m_data_encr_size;
	memcpy(mp_data_encr, &header, sizeof(header));
	return 0;
}

int AES::decrypt() {

	if (m_is_encr == -1) {
		cout << "No data imported yet" << endl;
		return -1;
	}
	else if (m_is_encr == 0) return -1;
	else {

		// The KeyExpansion routine must be called before decryption.
		keyExpansion();

		// decrypts data
		for (int i=0; i<m_nBlocks; ++i) {
			invCipher(i);
		}
		m_is_encr = 0;
	}
	return 0;
}

void AES::importData(char* p_data, unsigned int size) { // size = 0 für verschlüsselte daten

	bool data_is_encr = false;
	m_nBlocks = ((size-1)/16)+1;
	m_nBytes = m_nBlocks * 16;

	unsigned long min_data_decr_size = size;
	unsigned long min_data_encr_size = m_nBytes + sizeof(AESHeader);

	if (size >= sizeof(AESHeader)) {
		AESHeader header;
		memcpy(&header, p_data, sizeof(header));

		if (strncmp(header.uin,UIN, UINSIZE) == 0) { // data is encrypted

			min_data_encr_size = header.encr_size;
			min_data_decr_size = header.decr_size;
			data_is_encr = true;
		}
	}

	if (min_data_encr_size > m_data_encr_size || min_data_decr_size > m_data_decr_size) {
		if (m_is_encr != -1) {
			free(mp_data_encr);
			free(mp_data_decr);
		}
		mp_data_encr = (char*)malloc(min_data_encr_size);
		mp_data_decr = (char*)malloc(min_data_decr_size);
	}
	m_data_encr_size = min_data_encr_size;
	m_data_decr_size = min_data_decr_size;

	if (data_is_encr) {
		memcpy(mp_data_encr, p_data, m_data_encr_size);
		m_is_encr = 1;
	}
	else {
		memcpy(mp_data_decr, p_data, m_data_decr_size);
		m_is_encr = 0;
	}
}

char* AES::getData() {

	if ( m_is_encr == -1) {
		cout << "No data imported yet" << endl;
		return NULL;
	}
	else if ( m_is_encr == 0) return mp_data_decr;
	else if ( m_is_encr == 1) return mp_data_encr;
}


int AES::importDataFromFile(const std::string& fname) {

	std::ifstream ifile;
	ifile.open(fname.c_str(),std::ios::binary);

	//test if file is open
	if(ifile.is_open()) {
		// get size of file
		ifile.seekg(0, std::ios::end);

		unsigned long file_size = ifile.tellg();
		ifile.seekg(0, std::ios::beg);

		// allocate memory
		char* file_buf = (char*)malloc(file_size);
		// read data as a block
		ifile.read(file_buf,file_size);
		importData(file_buf, file_size);
		free(file_buf);

		ifile.close();
	}
	else {
		cout << "Error opening file " << fname << endl;
		return -1;
	}

	return 0;
}

int AES::writeDataToFile(const std::string& fname) {

	std::ofstream ofile;
	ofile.open(fname.c_str(),std::ios::binary);
	if(ofile.is_open()) {
		// write file
		if (m_is_encr) {
			ofile.write(getData(),m_data_encr_size);
		}
		else {
			ofile.write(getData(),m_data_decr_size);
		}

		// close file pointer
		ofile.close();
	}
	else {
		cout << "Error opening file " << fname << endl;
		return -1;
	}
	return 0;
}



// private methods

const unsigned char AES::m_pKey[] = {0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f};

const int AES::m_sbox[] = {
	//0     1    2      3     4    5     6     7      8    9     A      B    C     D     E     F
	0x63, 0x7c, 0x77, 0x7b, 0xf2, 0x6b, 0x6f, 0xc5, 0x30, 0x01, 0x67, 0x2b, 0xfe, 0xd7, 0xab, 0x76,		// 0
	0xca, 0x82, 0xc9, 0x7d, 0xfa, 0x59, 0x47, 0xf0, 0xad, 0xd4, 0xa2, 0xaf, 0x9c, 0xa4, 0x72, 0xc0,		// 1
	0xb7, 0xfd, 0x93, 0x26, 0x36, 0x3f, 0xf7, 0xcc, 0x34, 0xa5, 0xe5, 0xf1, 0x71, 0xd8, 0x31, 0x15,		// 2
	0x04, 0xc7, 0x23, 0xc3, 0x18, 0x96, 0x05, 0x9a, 0x07, 0x12, 0x80, 0xe2, 0xeb, 0x27, 0xb2, 0x75,		// 3
	0x09, 0x83, 0x2c, 0x1a, 0x1b, 0x6e, 0x5a, 0xa0, 0x52, 0x3b, 0xd6, 0xb3, 0x29, 0xe3, 0x2f, 0x84,		// 4
	0x53, 0xd1, 0x00, 0xed, 0x20, 0xfc, 0xb1, 0x5b, 0x6a, 0xcb, 0xbe, 0x39, 0x4a, 0x4c, 0x58, 0xcf,		// 5
	0xd0, 0xef, 0xaa, 0xfb, 0x43, 0x4d, 0x33, 0x85, 0x45, 0xf9, 0x02, 0x7f, 0x50, 0x3c, 0x9f, 0xa8,		// 6
	0x51, 0xa3, 0x40, 0x8f, 0x92, 0x9d, 0x38, 0xf5, 0xbc, 0xb6, 0xda, 0x21, 0x10, 0xff, 0xf3, 0xd2,		// 7
	0xcd, 0x0c, 0x13, 0xec, 0x5f, 0x97, 0x44, 0x17, 0xc4, 0xa7, 0x7e, 0x3d, 0x64, 0x5d, 0x19, 0x73,		// 8
	0x60, 0x81, 0x4f, 0xdc, 0x22, 0x2a, 0x90, 0x88, 0x46, 0xee, 0xb8, 0x14, 0xde, 0x5e, 0x0b, 0xdb,		// 9
	0xe0, 0x32, 0x3a, 0x0a, 0x49, 0x06, 0x24, 0x5c, 0xc2, 0xd3, 0xac, 0x62, 0x91, 0x95, 0xe4, 0x79,		// A
	0xe7, 0xc8, 0x37, 0x6d, 0x8d, 0xd5, 0x4e, 0xa9, 0x6c, 0x56, 0xf4, 0xea, 0x65, 0x7a, 0xae, 0x08,		// B
	0xba, 0x78, 0x25, 0x2e, 0x1c, 0xa6, 0xb4, 0xc6, 0xe8, 0xdd, 0x74, 0x1f, 0x4b, 0xbd, 0x8b, 0x8a,		// C
	0x70, 0x3e, 0xb5, 0x66, 0x48, 0x03, 0xf6, 0x0e, 0x61, 0x35, 0x57, 0xb9, 0x86, 0xc1, 0x1d, 0x9e,		// D
	0xe1, 0xf8, 0x98, 0x11, 0x69, 0xd9, 0x8e, 0x94, 0x9b, 0x1e, 0x87, 0xe9, 0xce, 0x55, 0x28, 0xdf,		// E
	0x8c, 0xa1, 0x89, 0x0d, 0xbf, 0xe6, 0x42, 0x68, 0x41, 0x99, 0x2d, 0x0f, 0xb0, 0x54, 0xbb, 0x16 };	// F

const int AES::m_rcon[] = {
	0x8d, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1b, 0x36, 0x6c, 0xd8, 0xab, 0x4d, 0x9a,
	0x2f, 0x5e, 0xbc, 0x63, 0xc6, 0x97, 0x35, 0x6a, 0xd4, 0xb3, 0x7d, 0xfa, 0xef, 0xc5, 0x91, 0x39,
	0x72, 0xe4, 0xd3, 0xbd, 0x61, 0xc2, 0x9f, 0x25, 0x4a, 0x94, 0x33, 0x66, 0xcc, 0x83, 0x1d, 0x3a,
	0x74, 0xe8, 0xcb, 0x8d, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1b, 0x36, 0x6c, 0xd8,
	0xab, 0x4d, 0x9a, 0x2f, 0x5e, 0xbc, 0x63, 0xc6, 0x97, 0x35, 0x6a, 0xd4, 0xb3, 0x7d, 0xfa, 0xef,
	0xc5, 0x91, 0x39, 0x72, 0xe4, 0xd3, 0xbd, 0x61, 0xc2, 0x9f, 0x25, 0x4a, 0x94, 0x33, 0x66, 0xcc,
	0x83, 0x1d, 0x3a, 0x74, 0xe8, 0xcb, 0x8d, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1b,
	0x36, 0x6c, 0xd8, 0xab, 0x4d, 0x9a, 0x2f, 0x5e, 0xbc, 0x63, 0xc6, 0x97, 0x35, 0x6a, 0xd4, 0xb3,
	0x7d, 0xfa, 0xef, 0xc5, 0x91, 0x39, 0x72, 0xe4, 0xd3, 0xbd, 0x61, 0xc2, 0x9f, 0x25, 0x4a, 0x94,
	0x33, 0x66, 0xcc, 0x83, 0x1d, 0x3a, 0x74, 0xe8, 0xcb, 0x8d, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20,
	0x40, 0x80, 0x1b, 0x36, 0x6c, 0xd8, 0xab, 0x4d, 0x9a, 0x2f, 0x5e, 0xbc, 0x63, 0xc6, 0x97, 0x35,
	0x6a, 0xd4, 0xb3, 0x7d, 0xfa, 0xef, 0xc5, 0x91, 0x39, 0x72, 0xe4, 0xd3, 0xbd, 0x61, 0xc2, 0x9f,
	0x25, 0x4a, 0x94, 0x33, 0x66, 0xcc, 0x83, 0x1d, 0x3a, 0x74, 0xe8, 0xcb, 0x8d, 0x01, 0x02, 0x04,
	0x08, 0x10, 0x20, 0x40, 0x80, 0x1b, 0x36, 0x6c, 0xd8, 0xab, 0x4d, 0x9a, 0x2f, 0x5e, 0xbc, 0x63,
	0xc6, 0x97, 0x35, 0x6a, 0xd4, 0xb3, 0x7d, 0xfa, 0xef, 0xc5, 0x91, 0x39, 0x72, 0xe4, 0xd3, 0xbd,
	0x61, 0xc2, 0x9f, 0x25, 0x4a, 0x94, 0x33, 0x66, 0xcc, 0x83, 0x1d, 0x3a, 0x74, 0xe8, 0xcb };

const int AES::m_rsbox[] = {
	0x52, 0x09, 0x6a, 0xd5, 0x30, 0x36, 0xa5, 0x38, 0xbf, 0x40, 0xa3, 0x9e, 0x81, 0xf3, 0xd7, 0xfb,
	0x7c, 0xe3, 0x39, 0x82, 0x9b, 0x2f, 0xff, 0x87, 0x34, 0x8e, 0x43, 0x44, 0xc4, 0xde, 0xe9, 0xcb,
	0x54, 0x7b, 0x94, 0x32, 0xa6, 0xc2, 0x23, 0x3d, 0xee, 0x4c, 0x95, 0x0b, 0x42, 0xfa, 0xc3, 0x4e,
	0x08, 0x2e, 0xa1, 0x66, 0x28, 0xd9, 0x24, 0xb2, 0x76, 0x5b, 0xa2, 0x49, 0x6d, 0x8b, 0xd1, 0x25,
	0x72, 0xf8, 0xf6, 0x64, 0x86, 0x68, 0x98, 0x16, 0xd4, 0xa4, 0x5c, 0xcc, 0x5d, 0x65, 0xb6, 0x92,
	0x6c, 0x70, 0x48, 0x50, 0xfd, 0xed, 0xb9, 0xda, 0x5e, 0x15, 0x46, 0x57, 0xa7, 0x8d, 0x9d, 0x84,
	0x90, 0xd8, 0xab, 0x00, 0x8c, 0xbc, 0xd3, 0x0a, 0xf7, 0xe4, 0x58, 0x05, 0xb8, 0xb3, 0x45, 0x06,
	0xd0, 0x2c, 0x1e, 0x8f, 0xca, 0x3f, 0x0f, 0x02, 0xc1, 0xaf, 0xbd, 0x03, 0x01, 0x13, 0x8a, 0x6b,
	0x3a, 0x91, 0x11, 0x41, 0x4f, 0x67, 0xdc, 0xea, 0x97, 0xf2, 0xcf, 0xce, 0xf0, 0xb4, 0xe6, 0x73,
	0x96, 0xac, 0x74, 0x22, 0xe7, 0xad, 0x35, 0x85, 0xe2, 0xf9, 0x37, 0xe8, 0x1c, 0x75, 0xdf, 0x6e,
	0x47, 0xf1, 0x1a, 0x71, 0x1d, 0x29, 0xc5, 0x89, 0x6f, 0xb7, 0x62, 0x0e, 0xaa, 0x18, 0xbe, 0x1b,
	0xfc, 0x56, 0x3e, 0x4b, 0xc6, 0xd2, 0x79, 0x20, 0x9a, 0xdb, 0xc0, 0xfe, 0x78, 0xcd, 0x5a, 0xf4,
	0x1f, 0xdd, 0xa8, 0x33, 0x88, 0x07, 0xc7, 0x31, 0xb1, 0x12, 0x10, 0x59, 0x27, 0x80, 0xec, 0x5f,
	0x60, 0x51, 0x7f, 0xa9, 0x19, 0xb5, 0x4a, 0x0d, 0x2d, 0xe5, 0x7a, 0x9f, 0x93, 0xc9, 0x9c, 0xef,
	0xa0, 0xe0, 0x3b, 0x4d, 0xae, 0x2a, 0xf5, 0xb0, 0xc8, 0xeb, 0xbb, 0x3c, 0x83, 0x53, 0x99, 0x61,
	0x17, 0x2b, 0x04, 0x7e, 0xba, 0x77, 0xd6, 0x26, 0xe1, 0x69, 0x14, 0x63, 0x55, 0x21, 0x0c, 0x7d };

int AES::getSBoxValue(int num) {

	return m_sbox[num];
}

int AES::getSBoxInvert(int num) {

	return m_rsbox[num];
}

void AES::keyExpansion() {

	int i,j;
	unsigned char temp[4],k;

	// First round key is the key itself.
	for(i=0;i<m_nk;i++) {
		m_roundkey[i*4]   = m_pKey[i*4];
		m_roundkey[i*4+1] = m_pKey[i*4+1];
		m_roundkey[i*4+2] = m_pKey[i*4+2];
		m_roundkey[i*4+3] = m_pKey[i*4+3];
	}	// i = m_nk

	// All other round keys are found from the previous round keys.
	while (i < (Nb * (m_nr+1))) {

		for(j=0;j<4;j++) {
			temp[j]=m_roundkey[(i-1) * 4 + j];
		}

		if (i % m_nk == 0) {

			// This rotates the 4 bytes in a word to the left once ([a0,a1,a2,a3] --> [a1,a2,a3,a0])
			{
				k = temp[0];
				temp[0] = temp[1];
				temp[1] = temp[2];
				temp[2] = temp[3];
				temp[3] = k;
			}

			// This takes a four-byte input word and applies the S-Box to each of the four bytes to produce an output word.
			{
				temp[0] = getSBoxValue(temp[0]);
				temp[1] = getSBoxValue(temp[1]);
				temp[2] = getSBoxValue(temp[2]);
				temp[3] = getSBoxValue(temp[3]);
			}

			temp[0] =  temp[0] ^ m_rcon[i/m_nk];
		}

		else if (m_nk > 6 && i % m_nk == 4) {

			{
				temp[0] = getSBoxValue(temp[0]);
				temp[1] = getSBoxValue(temp[1]);
				temp[2] = getSBoxValue(temp[2]);
				temp[3] = getSBoxValue(temp[3]);
			}
		}

			m_roundkey[i*4+0] = m_roundkey[(i-m_nk)*4+0] ^ temp[0];
			m_roundkey[i*4+1] = m_roundkey[(i-m_nk)*4+1] ^ temp[1];
			m_roundkey[i*4+2] = m_roundkey[(i-m_nk)*4+2] ^ temp[2];
			m_roundkey[i*4+3] = m_roundkey[(i-m_nk)*4+3] ^ temp[3];
			i++;
		}
}

void AES::addRoundKey(int round) {

	int i,j;
	for(i=0;i<4;i++) {
		for(j=0;j<4;j++) {
			m_state[j][i] ^= m_roundkey[round * Nb * 4 + i * Nb + j];
		}
	}
}

void AES::subBytes() {

	int i,j;
	for(i=0;i<4;i++) {
		for(j=0;j<4;j++) {
			m_state[i][j] = getSBoxValue(m_state[i][j]);
		}
	}
}

void AES::invSubBytes() {

	int i,j;
	for(i=0;i<4;i++) {
		for(j=0;j<4;j++) {
			m_state[i][j] = getSBoxInvert(m_state[i][j]);

		}
	}
}

void AES::shiftRows() {

	// This shifts ach row with a different offset (offset = number of row)

	unsigned char temp;

	// Rotate first row 1 columns to left
	temp=m_state[1][0];
	m_state[1][0] = m_state[1][1];
	m_state[1][1] = m_state[1][2];
	m_state[1][2] = m_state[1][3];
	m_state[1][3] = temp;

	// Rotate second row 2 columns to left
	temp = m_state[2][0];
	m_state[2][0] = m_state[2][2];
	m_state[2][2] = temp;

	temp = m_state[2][1];
	m_state[2][1] = m_state[2][3];
	m_state[2][3] = temp;

	// Rotate third row 3 columns to left
	temp = m_state[3][0];
	m_state[3][0] = m_state[3][3];
	m_state[3][3] = m_state[3][2];
	m_state[3][2] = m_state[3][1];
	m_state[3][1] = temp;

}

void AES::invShiftRows() {

	unsigned char temp;

	// Rotate first row 1 columns to right
	temp = m_state[1][3];
	m_state[1][3] = m_state[1][2];
	m_state[1][2] = m_state[1][1];
	m_state[1][1] = m_state[1][0];
	m_state[1][0] = temp;

	// Rotate second row 2 columns to right
	temp = m_state[2][0];
	m_state[2][0] = m_state[2][2];
	m_state[2][2] = temp;

	temp = m_state[2][1];
	m_state[2][1] = m_state[2][3];
	m_state[2][3] = temp;

	// Rotate third row 3 columns to right
	temp = m_state[3][0];
	m_state[3][0] = m_state[3][1];
	m_state[3][1] = m_state[3][2];
	m_state[3][2] = m_state[3][3];
	m_state[3][3] = temp;

}

void AES::mixColumns() {

	int i;
	unsigned char Tmp,Tm,t;
	for(i=0;i<4;i++)
	{
		t = m_state[0][i];
		Tmp = m_state[0][i] ^ m_state[1][i] ^ m_state[2][i] ^ m_state[3][i] ;
		Tm  = m_state[0][i] ^ m_state[1][i] ; Tm = xtime(Tm); m_state[0][i] ^= Tm ^ Tmp ;
		Tm  = m_state[1][i] ^ m_state[2][i] ; Tm = xtime(Tm); m_state[1][i] ^= Tm ^ Tmp ;
		Tm  = m_state[2][i] ^ m_state[3][i] ; Tm = xtime(Tm); m_state[2][i] ^= Tm ^ Tmp ;
		Tm  = m_state[3][i] ^ t ; Tm = xtime(Tm); m_state[3][i] ^= Tm ^ Tmp ;  // see macros for xtime(x)
	}

}

void AES::invMixColumns() {

	int i;
	unsigned char a,b,c,d;
	for(i=0;i<4;i++) {

		a = m_state[0][i];
		b = m_state[1][i];
		c = m_state[2][i];
		d = m_state[3][i];

		// see macros for Multiply(x,y)
		m_state[0][i] = Multiply(a, 0x0e) ^ Multiply(b, 0x0b) ^ Multiply(c, 0x0d) ^ Multiply(d, 0x09);
		m_state[1][i] = Multiply(a, 0x09) ^ Multiply(b, 0x0e) ^ Multiply(c, 0x0b) ^ Multiply(d, 0x0d);
		m_state[2][i] = Multiply(a, 0x0d) ^ Multiply(b, 0x09) ^ Multiply(c, 0x0e) ^ Multiply(d, 0x0b);
		m_state[3][i] = Multiply(a, 0x0b) ^ Multiply(b, 0x0d) ^ Multiply(c, 0x09) ^ Multiply(d, 0x0e);
	}
}

void AES::cipher(unsigned int block) {

	int i,j,round=0;
	char* mp_pure_data_encr = mp_data_encr + sizeof(AESHeader);

	// Copy the input data to state array.
	for(i=0;i<4;i++) {
		for(j=0;j<4;j++) {
			m_state[j][i] = mp_data_decr[i*4 + j + block*16];
		}
	}

	// Add the First round key to state before starting the rounds.
	addRoundKey(0);

	// There will be m_nr rounds, whereas the first m_nr-1 rounds are identical.
	for(round=1;round<m_nr;round++) {
		subBytes();
		shiftRows();
		mixColumns();
		addRoundKey(round);
	}

	// No MixColums in last round
	subBytes();
	shiftRows();
	addRoundKey(m_nr);

	// Copy encrypted data from state array to Output array.
	for(i=0;i<4;i++) {
		for(j=0;j<4;j++) {
			mp_pure_data_encr[i*4 + j + block*16] = m_state[j][i];
		}
	}

}

void AES::invCipher(unsigned int block) {

	int i,j,round = 0;
	char* mp_pure_data_encr = mp_data_encr + sizeof(AESHeader);

	//Copy the input data to state array.
	for(i=0;i<4;i++) {
		for(j=0;j<4;j++) {
			m_state[j][i] = mp_pure_data_encr[i*4 + j + block*16];
		}
	}

	// Add the First round key to state before starting the rounds.
	addRoundKey(m_nr);

	// There will be m_nr rounds, whereas the first m_nr-1 rounds are identical.
	for(round=m_nr-1;round>0;round--) {
		invShiftRows();
		invSubBytes();
		addRoundKey(round);
		invMixColumns();
	}

	// No MixColums in last round
	invShiftRows();
	invSubBytes();
	addRoundKey(0);

	// Copy decrypted data from state array to Output array.
	for(i=0;i<4;i++) {
		for(j=0;j<4;j++) {
			mp_data_decr[i*4 + j + block*16]=m_state[j][i];
		}
	}

}


