// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  AES.h
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Stephan Riedel, stephan-riedel@gmx.de
// Date:      30.05.2008
// *****************************************************************

#ifndef _AES_H_
#define _AES_H_

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <iostream>
#include <fstream>

#define UIN "3141592653"
#define UINSIZE 10
#define KEYSIZE 16

struct AESHeader {
	char uin[UINSIZE+1];			//!< id number to identify encr. data
	unsigned int decr_size;			//!< size of decr. data
	unsigned int encr_size;			//!< size of encr. data
};

//! Class for handling AES encryption and decryption.
class AES {

public:
	AES();
	~AES();

	//! Encrypts the data. Returns -1 if there is an error. 0 otherwise.
	int encrypt();
	//! Decrypts the data. Returns -1 if there is an error. 0 otherwise.
	int decrypt();

	void importData(char* p_data, unsigned int size);
	char* getData();
	char* const getData()  const { return getData(); }
	int getEncryptedSize() const { return m_data_encr_size; }
	int getDecryptedSize() const { return m_data_decr_size; }

	int importDataFromFile(const std::string& fname);
	int writeDataToFile(const std::string& fname);

	int isEncrypted() const { return m_is_encr; }
	unsigned long decrToEncrSize(unsigned long size) { return (((size-1)/16)+1) * 16 + sizeof(AESHeader); }

private:
	char* mp_data_encr;
	char* mp_data_decr;
	int m_is_encr;					//!< encryption status
	unsigned long m_data_encr_size;
	unsigned long m_data_decr_size;

	void keyExpansion();										//!< Produces Nb(Nr+1) round keys.

	// Encryption:
	int getSBoxValue(int num);									//!< Returns S-Box value
	void addRoundKey(int round);								//!< Adds the round key to state by an XOR function
	void subBytes();											//!< Substitutes the values in the state matrix with values in an S-Box
	void shiftRows();											//!< Shifts the rows in state matrix to the left
	void mixColumns();											//!< Mixes the columns of the state matrix
	void cipher(unsigned int block);							//!< Main function that encrypts the PlainText

	// Decryption:
	int getSBoxInvert(int num);									//!< Returns value of invert S-Box
	void invSubBytes();											//!< Inversive SubBytes function
	void invShiftRows();  										//!< Inversive ShiftRows function
	void invMixColumns();										//!< Inversive MixColums function
	void invCipher(unsigned int block);							//!< Main function that decrypts the CipherText

	unsigned int m_data_size;									//!< input data size
	unsigned int m_key_size;									//!< key size
	unsigned int m_nBlocks;										//!< Number of blocks to be ciphered
	unsigned int m_nBytes;										//!< Number of bytes to be ciphered
	int m_nr;													//!< Number of rounds in AES Cipher
	int m_nk;													//!< Number of 32 bit words in key
	unsigned char m_roundkey[240];								//!< Stores the round keys
	unsigned char m_state[4][4];								//!< intermediate results during encryption
	static const int m_sbox[];									//!< S-Box
	static const int m_rcon[];									//!< Rcon
	static const int m_rsbox[];									//!< RS-Box
	static const unsigned char m_pKey[];						//!< Stores the key for the en- & decryption

};

#endif /* _AES_H_ */


