// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  AbstractDatagram.h
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder
// Date:      03.11.2008
// *****************************************************************

#ifndef _ABSTRACTDATAGRAM_H_
#define _ABSTRACTDATAGRAM_H_

#include <sys/types.h>
#include <netinet/in.h>

typedef struct {
	u_int16_t id;					//!< header id
	u_int16_t size;					//!< size of entire message
	u_int32_t count;				//!< consecutive number
	u_int8_t has_network_format;	//!< current state of byte order format
} __attribute__ ((packed)) datagram_header_t;

//! Abstract class representing an UDP datagram.
class AbstractDatagram {

public:

	//! Constructor that checks for little-endianess
	AbstractDatagram();

	//! Virtual destructor for supporting polymorphism. Sub-classes should also define a virtual destructor.
	virtual ~AbstractDatagram();

	/*!
	 * Initializes the header structure and other internal members of AbstractDatagram.
	 * Needs to be called in a sub-class -- e.g. in the constructor of a sub-class -- to initialize
	 * the header of the datagram.
	 *
	 * @param h   Points to the header of this datagram.
	 * @param s   The size of the whole datagram.
	 * @param id  An unique ID that identifies this datagram type.
	 */
	void initDatagram(datagram_header_t* h, u_int16_t s, int id);

	//! Returns true if this platform is a little endian platform.
	static bool checkLittleEndian();

	//! Abstract function that sub-classes implement to do the conversion from network to host encoding.
	/*!
	 * In this function only the data of the datagram needs to be converted with ntohs, nothl etc.
	 */
	virtual void ntohData() = 0;

	//! Abstract function that sub-classes implement to do the conversion from host to network encoding.
	/*!
	 * In this function only the data of the datagram needs to be converted with htons, hotnl etc.
	 */
	virtual void htonData() = 0;

	//! Function that does the conversion from network to host for the header. Does not need to be reimplemented in sub-classes.
	static void ntohHeader(datagram_header_t* h);

	//! Function that does the conversion from host to network for the header. Does not need to be reimplemented in sub-classes.
	static void htonHeader(datagram_header_t* h);

	//! This function calls the conversion functions to convert from network to host if it is necessary.
	void ntoh();

	//! This function calls the conversion functions to convert from host to network if it is necessary.
	void hton();

	//! Returns an untyped pointer to the whole datagram (header and data).
	char* getData() { return (char*)header_p; }

	//! Returns an untyped const pointer to the whole datagram (header and data).
	char* const getData() const { return (char*)header_p; }

	//! Returns the ID of this datagram.
	u_int16_t getId() const { return header_p->id; }

	//! Returns the size of this datagram. That is the size of the header and the data.
	u_int16_t getSize() const;

	//! Returns the value of the counter.
	u_int32_t getCount() const { return header_p->count; }

	//! Returns true when the datagram is in network format. false otherwise.
	u_int8_t  hasNetworkFormat() const { return header_p->has_network_format; }

	//! Increments the value of the counter.
	void incCount() { ++(header_p->count); }

private:
	static bool isLE;
	bool isLittleEndian() const { return isLE; }
	datagram_header_t* header_p;
	u_int16_t msg_size;	//!< msg size, this is needed for send routine and does not need conversion, always host format!
};

#endif /* _ABSTRACTDATAGRAM_H_ */


