// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  Mutex.h
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder, Daniel Jagszent
// Date:      26.04.2008
// *****************************************************************

#ifndef _MUTEX_H_
#define _MUTEX_H_

#include <pthread.h>

using namespace std;

//! C++-Wrapper for pthread-mutexes.
class Mutex {
public:
	Mutex();
	~Mutex();

	//! Waits until the current thread got the mutex. Do not call this method more than once from the same thread without calling unlock() in between.
	void lock();
	//! Releases the lock on the mutex. You have to ensure that the current thread has gotten a lock the mutex first.
	void unlock();

private:
	Mutex ( const Mutex& );	//!< disabled copy-constructor
	pthread_mutex_t mutex;
	friend class WaitCondition;
};


//! This class gets a lock on a Mutex for its whole life-time.
class MutexLocker {

public:
	//! Constructor gets a lock on Mutex m.
	MutexLocker(Mutex& m) : mutex(m), locked(true) { mutex.lock(); }
	//! Destructor releases the lock on the Mutex if it is not already done.
	~MutexLocker() { unlock(); }
	//! Manually releases the lock on the Mutex before the object gets out of scope.
	void unlock() { if (locked) { mutex.unlock(); locked = false; } }
private:
	MutexLocker ( const MutexLocker& );	//!< disabled copy-constructor
	void operator=(const MutexLocker&);
	Mutex& mutex;
	bool locked;
};

#endif /* _MUTEX_H_ */
