// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  PeriodicThread.cpp
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder, Daniel Jagszent
// Date:      21.05.2008
// *****************************************************************

#include "PeriodicThread.h"

#include "timing_functions.h"

#define CLOCK_SOURCE CLOCK_REALTIME

PeriodicThread::PeriodicThread(long long period, wait_type_t wait_type) :
	m_period(period), m_wait_type(wait_type), m_abort(false), m_paused(false) {
	timespec timer_resolution;
	clock_getres(CLOCK_SOURCE, &timer_resolution);
	m_timer_resolution = getNanoseconds(timer_resolution);
}

PeriodicThread::~PeriodicThread() {
}

void PeriodicThread::terminate() {
	m_abort = true;
}
void PeriodicThread::resume() {
	m_paused = false;
}
void PeriodicThread::pause() {
	m_paused = true;
}
void PeriodicThread::init() {
	getCurrentTime(m_start);
	m_next_timeout = m_start;
	m_next_timeout.tv_nsec += m_period;
	normalizeTimespec(m_next_timeout);
}

void PeriodicThread::wait_for_next_turn() {
	long long wait_time;
	if (m_wait_type == WAIT_ABS_NANOSLEEP) {
		wait_time = m_period;
	} else {
		timespec now;
		wait_time = m_period - (getNanoseconds(getCurrentTime(now)) - getNanoseconds(m_start));
		if (wait_time < 0) { // we missed a period
			wait_time = m_period;
		}
	}
	// Subtract timer resolution from wait time so that we can busy wait the last fraction
	if (m_wait_type == WAIT_NANOSLEEP_AND_BUSYWAIT) {
		wait_time -= m_timer_resolution;
	}
	if (wait_time > 0) {
		timespec req, remain;
		if (m_wait_type == WAIT_ABS_NANOSLEEP) {
			req = m_next_timeout;
		} else {
			req.tv_sec = 0;
			req.tv_nsec = wait_time;
			normalizeTimespec(req);
		}
		int rc = clock_nanosleep(CLOCK_SOURCE, m_wait_type==WAIT_ABS_NANOSLEEP?TIMER_ABSTIME:0, &req, &remain);
		while (rc == EINTR) { // clock_nanosleep can wake up prematurely - if that happens, just call it again
			if (m_wait_type != WAIT_ABS_NANOSLEEP) {
				req = remain; // remain holds the remaining time to wait. Makes only sense for !TIMER_ABSTIME
			}
			rc = clock_nanosleep(CLOCK_SOURCE, m_wait_type==WAIT_ABS_NANOSLEEP?TIMER_ABSTIME:0, &req, &remain);
		}
		if (rc != 0) { // if there's an error, throw an exception
			errno = rc;
			throw Exception("PeriodicThread::wait_for_next_turn(): Error while nanosleeping");
		}
	}
	// wait the last m_timer_resolution nanoseconds
	// by actively polling the current time and compare it with the desired timeout time
	if (m_wait_type == WAIT_NANOSLEEP_AND_BUSYWAIT) {
		while (getCurrentTime(m_start) < m_next_timeout)
			;
	}

	getCurrentTime(m_start);
	m_next_timeout.tv_nsec += m_period;
	normalizeTimespec(m_next_timeout);
}

void PeriodicThread::run() {
	init();
	while (!m_abort) {
		if (!m_paused) {
			do_turn();
		}
		wait_for_next_turn();
	}
}
//
//void NanosleepPeriodicThread::init() {
//	getCurrentTime(m_start);
//}
//void NanosleepPeriodicThread::wait_for_next_turn() {
//	long long start_time = getNanoseconds(m_start);
//	timespec now;
//	long long now_time = getNanoseconds(getCurrentTime(now));
//	long long wait_time = m_period - (now_time - start_time);
//	if (wait_time < 0) { // we missed a period
//		wait_time = m_period;
//	}
//	timespec req, remain;
//	req.tv_sec = 0;
//	req.tv_nsec = wait_time;
//	normalizeTimespec(req);
//	while (nanosleep(&req, &remain) < 0) {
//		if (errno== EINTR) { // nanosleep can be interrupted
//			req = remain; // if it does, just wait for the remaining time
//		} else {
//			throw Exception(
//					"NanosleepPeriodicThread::wait_for_next_turn(): Error while nanosleeping");
//		}
//	}
//	getCurrentTime(m_start);
//}
//
//void NanosleepPollingPeriodicThread::init() {
//	timespec timer_resolution;
//	clock_getres(CLOCK_REALTIME, &timer_resolution);
//	m_timer_resolution = getNanoseconds(timer_resolution);
//	getCurrentTime(m_start);
//}
//void NanosleepPollingPeriodicThread::wait_for_next_turn() {
//	long long start_time = getNanoseconds(m_start);
//	timespec now, timeout;
//	long long now_time = getNanoseconds(getCurrentTime(now));
//	long long wait_time = m_period - (now_time - start_time);
//	if (wait_time < 0) { // we missed a period
//		wait_time = m_period;
//	}
//	wait_time -= m_timer_resolution;
//	if (wait_time > 0) {
//		timespec req, remain;
//		req.tv_sec = 0;
//		req.tv_nsec = wait_time;
//		normalizeTimespec(req);
//		while (nanosleep(&req, &remain) < 0) {
//			if (errno== EINTR) { // nanosleep can be interrupted
//				req = remain; // if it does, just wait for the remaining time
//			} else {
//				throw Exception(
//						"NanosleepPollingPeriodicThread::wait_for_next_turn(): Error while nanosleeping");
//			}
//		}
//	}
//	timeout = now;
//	timeout.tv_nsec += m_period;
//	normalizeTimespec(timeout);
//	while (getCurrentTime(m_start) < timeout)
//		;
//}
//
//void AbsNanosleepPeriodicThread::init() {
//	getCurrentTime(m_next_timeout);
//	m_next_timeout.tv_nsec += m_period;
//	normalizeTimespec(m_next_timeout);
//}
//void AbsNanosleepPeriodicThread::wait_for_next_turn() {
//	int rc = clock_nanosleep(CLOCK_SOURCE, TIMER_ABSTIME, &m_next_timeout, 0);
//	while ( rc == EINTR ){
//		;
//	}
//	m_next_timeout.tv_nsec += m_period;
//	normalizeTimespec(m_next_timeout);
//}

