// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  PeriodicThread.h
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder, Daniel Jagszent
// Date:      21.05.2008
// *****************************************************************

#ifndef PERIODICTHREAD_H_
#define PERIODICTHREAD_H_

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>

#include "Thread.h"

#ifndef DEFAULT_WAIT_TYPE
//! This macro can get overridden *before* you include PeriodicThread.h to change the default wait type.
#define DEFAULT_WAIT_TYPE WAIT_ABS_NANOSLEEP
#endif

//! A Thread sub-class that handles periodic events with a fixed interval.
class PeriodicThread: public Thread {
public:
	//! The method that should be used for waiting for the next period to begin.
	enum wait_type_t { WAIT_NANOSLEEP, WAIT_NANOSLEEP_AND_BUSYWAIT, WAIT_ABS_NANOSLEEP };
	//! The constructor needs the interval length (in nanoseconds) and the method of waiting as parameters.
	PeriodicThread(long long period, wait_type_t wait_type = DEFAULT_WAIT_TYPE);
	virtual ~PeriodicThread();
	//! Gracefully aborts the Thread after the current period is done.
	void terminate();
	//! Pauses the execution. The waiting for periods is not paused, though.
	void pause();
	//! Resumes the execution. Calling pause() and resume() does not changes the interval nor the point in time when the execution is performed.
	void resume();
protected:
	//! You should not need to override this function in sub-classes.
	virtual void run();
	//! You may override this method to implement initializations that need to be done in the execution thread. Be sure to call PeriodicThread::init, though.
	virtual void init();
	//! You should not need to override this function in sub-classes.
	virtual void wait_for_next_turn();
	//! This abstract method needs to be implemented in sub-classes. It gets called once for each period.
	virtual void do_turn() = 0;
protected:
	long long m_period;
	wait_type_t m_wait_type;
	timespec m_start;
	long long m_timer_resolution;
	timespec m_next_timeout;
	bool m_abort;
	bool m_paused;
};

#endif /* PERIODICTHREAD_H_ */
