// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  Socket.h
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder
// Date:      26.04.2008
// *****************************************************************

#ifndef _SOCKET_H_
#define _SOCKET_H_

#include <string>            // For string
#include <exception>         // For exception class

//! Signals a problem with the execution of a socket call.
class SocketException : public std::exception {
public:
	/*! Construct a SocketException with a explanatory message.
	 * should be postfixed to the user provided message */
	SocketException(const std::string &message, bool inclSysMsg = false) throw();


	//! Provided just to guarantee that no exceptions are thrown.
	~SocketException() throw();

	//! Get the exception message @return exception message
	const char *what() const throw();

private:
	std::string userMessage;  //!< Exception message
};

//! Base class representing basic communication endpoint
class Socket {
public:
	//! Close and deallocate this socket
	~Socket();

	//!Get the local address @return local address of socket @exception SocketException thrown if fetch fails
	std::string getLocalAddress() throw(SocketException);

	//! Get the local port
	unsigned short getLocalPort() throw(SocketException);

	//! Set the local port to the specified port and the local address to any interface
	void setLocalPort(unsigned short localPort) throw(SocketException);

	/*! Set the local port to the specified port and the local address
	 * to the specified address.  If you omit the port, a random port
	 * will be selected. */
	void setLocalAddressAndPort(const std::string &localAddress,
			unsigned short localPort = 0) throw(SocketException);

	static void cleanUp() throw(SocketException);

	//! Resolve the specified service for the specified protocol to the corresponding port number in host byte order
	static unsigned short resolveService(const std::string &service,
			const std::string &protocol = "tcp");

	//--------------

	//! Establish a socket connection with the given foreign address and port
	void connect(const std::string &foreignAddress, unsigned short foreignPort)
	throw(SocketException);

	//! Write the given buffer to this socket.  Call connect() before calling send()
	void send(const void *buffer, int bufferLen) throw(SocketException);

	//! Read into the given buffer up to bufferLen bytes data from this socket.  Call connect() before calling recv()
	int recv(void *buffer, int bufferLen) throw(SocketException);

	//! Get the foreign address.  Call connect() before calling recv()
	std::string getForeignAddress() throw(SocketException);

	//! Get the foreign port.  Call connect() before calling recv()
	unsigned short getForeignPort() throw(SocketException);

private:
	//! Prevent the user from trying to use value semantics on this object
	Socket(const Socket &sock);
	void operator=(const Socket &sock);

protected:
	int sockDesc;              //!< Socket descriptor
	Socket(int type, int protocol) throw(SocketException);
	Socket(int sockDesc);
};

//! UDP socket class
class UDPSocket : public Socket {
public:

	//! Construct a UDP socket
	UDPSocket() throw(SocketException);

	//! Construct a UDP socket with the given local port
	UDPSocket(unsigned short localPort) throw(SocketException);

	//! Construct a UDP socket with the given local port and address
	UDPSocket(const std::string &localAddress, unsigned short localPort) throw(SocketException);

	//! Unset foreign address and port
	void disconnect() throw(SocketException);

	//! Send the given buffer as a UDP datagram to the specified address/port
	void sendTo(const void *buffer, int bufferLen, const std::string &foreignAddress,
			unsigned short foreignPort) throw(SocketException);

	//! Read read up to bufferLen bytes data from this socket.  The given buffer is where the data will be placed
	int recvFrom(void *buffer, int bufferLen, std::string &sourceAddress,
			unsigned short &sourcePort) throw(SocketException);

	//! Reads bufferLen bytes from this socket without consuming them.
	int recvFromPeek(void *buffer, int bufferLen, std::string &sourceAddress,
			unsigned short &sourcePort) throw(SocketException);

private:
	void setBroadcast();
};

#endif /* _SOCKET_H_ */
