// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  Thread.cpp
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder, Daniel Jagszent
// Date:      26.04.2008
// *****************************************************************

#include "Thread.h"
#include <cerrno>
#include <string.h>
#include <sched.h>
#include <signal.h>
#include <iostream>

const int USED_RT_SCHEDULER_STRATEGY = SCHED_FIFO; // or SCHED_RR

Thread::Exception::Exception(const std::string& message) throw () :
  m_message(message) {
  m_message.append(": ");
  m_message.append(strerror(errno));
}
Thread::Exception::~Exception() throw () {
}
const char* Thread::Exception::what() const throw () {
  return m_message.c_str();
}


void Thread::setTaskPrio(Thread::Prio prio) throw(Exception)
{
  sched_param param;
  param.sched_priority = convertPrio(prio);
  if (sched_setscheduler(getpid(), SCHED_FIFO, &param) != 0) {
    throw Exception("Thread::setProcessPrio: Cannot set my scheduler & priority. You probably have to call this program as root!");
  }
}

void Thread::start(Thread::Prio prio) throw(Thread::Exception) {

  if (prio == PRIO_NO) { // normal SCHED_OTHER Thread
    if (pthread_create(&thread_id, 0, thread_starter, (void *) this) != 0) {
      throw Exception("Thread::start: Cannot create thread");
    }
  } else { // REAL-TIME Thread
    pthread_attr_t attr;
    sched_param param;
    if (pthread_attr_init(&attr) != 0) {
      throw Exception("Thread::start: Cannot initialize attribute object");
    }
    if (pthread_attr_setschedpolicy(&attr, USED_RT_SCHEDULER_STRATEGY) != 0) {
      throw Exception("Thread::start: Cannot set scheduling policy");
    }
    param.sched_priority = convertPrio(prio);
    if (pthread_attr_setschedparam(&attr, &param) != 0) {
      throw Exception("Thread::start: Cannot set scheduling parameter (priority)");
    }
    if (pthread_create(&thread_id, &attr, thread_starter, (void *) this) != 0) {
      pthread_attr_destroy(&attr);
      throw Exception("Thread::start: Cannot create thread");
    }
    if (pthread_attr_destroy(&attr) != 0) {
      throw Exception("Thread::start: Cannot destroy the attribute object");
    }
  }
}

bool Thread::wait(unsigned long time) const {
  if (time == ULONG_MAX) {
    if (pthread_join(thread_id, NULL) != 0) {
      throw Exception("Thread::wait: Cannot join thread");
    }
    return true;
  }
  else {
    while (time > 0) {
      if (!isrunning) break;
      time -= 10;
      usleep(10);
    }
    return time > 0;
  }
}

int Thread::convertPrio(Prio prio)
{
  static int max_prio = sched_get_priority_max(USED_RT_SCHEDULER_STRATEGY);
  static int min_prio = sched_get_priority_min(USED_RT_SCHEDULER_STRATEGY);
  static int normal_prio = min_prio + (max_prio - min_prio) / 2;
  switch (prio) {
    case PRIO_LOW: return min_prio;
    case PRIO_NORMAL: return normal_prio;
    case PRIO_HIGH: return max_prio;
    default: return -1;
  }
}

void* Thread::thread_starter(void* thread_obj) {
  Thread* self = reinterpret_cast<Thread*> (thread_obj);
  self->setRunning(true);
  self->run();
  self->setRunning(false);
  return 0;
}

