// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  Thread.h
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder, Daniel Jagszent
// Date:      26.04.2008
// *****************************************************************

#ifndef _THREAD_H_
#define _THREAD_H_

#include <stdexcept>
#include <pthread.h>
#include <limits.h>

//! C++-Thread class wrapping pthread C-interface interface.
class Thread {
public:

	//! Gets thrown when underlying pthread functions return an error.
	class Exception : public std::exception {
	public:
		explicit Exception(const std::string& message="") throw ();
		~Exception() throw ();
		virtual const char* what () const throw ();
	protected:
		std::string m_message;
	};

	//! The priority of a thread. If you assign a priority other than PRIO_NO this thread is considered a real-time thread and your program needs to be linked with the rt library.
	enum Prio { PRIO_NO, PRIO_LOW, PRIO_NORMAL, PRIO_HIGH };


public:
	virtual ~Thread() {};

	void start(Prio prio = PRIO_NO) throw(Exception);

	/*! Blocks the thread until either of these conditions is met:

	The thread has finished execution or was not started yet, return value is true in this case.
	time milliseconds has elapsed, returns false in this case. If time is ULONG_MAX (the default),
	then the wait will never timeout.

	*/
	bool wait(unsigned long time = ULONG_MAX) const;
	bool isRunning() const { return isrunning; }
	bool isFinished () const { return !isrunning; }
	void terminate() {}

	static void setTaskPrio(Prio prio = PRIO_NORMAL) throw(Exception);

protected:
	virtual void run() = 0;

private:
	pthread_t thread_id;
	bool isrunning;
	static void* thread_starter(void* thread_obj);
	void setRunning(bool newVal) { isrunning = newVal; }
	static int convertPrio(Prio prio);
};

#endif /* _THREAD_H_ */
