// *****************************************************************
// This file is part of the book "Embedded Linux - Das Praxisbuch"
//
// Copyright (C) 2008-2012 Joachim Schroeder
// Chair Prof. Dillmann (IAIM),
// Institute for Computer Science and Engineering,
// University of Karlsruhe. All rights reserved.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free
// Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.
// *****************************************************************

// *****************************************************************
// Filename:  iowarrior_i2c.c
// Copyright: Joachim Schroeder, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:    Joachim Schroeder
// Date:      01.09.2008
// *****************************************************************

#include "iowarrior_i2c.h"

void iow_print_warriors(const char* basename){

	int i, fd;
	int fd_count = 0;
	struct iowarrior_info info;
	char devname[25];

  	printf("----------------------\n");
	
	for(i = 0; i < MAXDEVICES * 2; i++) {
	
		sprintf(devname, "%s%d",basename, i);

	  	if(( fd = open(devname, O_RDWR)) < 0 )
    	continue;
    	
    	fd_count++;
    	
    	printf("Found warrior at device %s : \n", devname);
    	if( ioctl( fd, IOW_GETINFO, &info))
			printf( "Unable to retrieve device info" );
		else {
			printf("VendorId=%04x\n",info.vendor);
    		
    		printf("ProductId=%04x ",info.product);
    		if(info.product==USB_DEVICE_ID_CODEMERCS_IOW40)
				printf("(IOWarrior40)\n");
    		else if(info.product==USB_DEVICE_ID_CODEMERCS_IOW24)
					printf("(IOWarrior24)\n");
    		else if(info.product==USB_DEVICE_ID_CODEMERCS_IOW56)
      				printf("(IOWarrior56)\n");
    		else
				printf("(Unknown device!)\n");
    
    		printf("Serial=0x%s\nRevision=0x%04x\n",info.serial, info.revision); 
			if(info.if_num!=0)
				printf("Special Mode Interface\n");
			else
				printf("I/O Interface\n");
		}
    	close(fd);
    	printf("----------------------\n");
    }	
    printf("%d Devices found\n", fd_count);
}

int iow_find_special_descriptor(const char *serial, const char* basename) {

	char devname[25];
	struct iowarrior_info info;
	int i, fd;

	for(i = 0; i < MAXDEVICES * 2; i++) {
	
		sprintf(devname, "%s%d",basename, i);
	  	if(( fd = open(devname, O_RDWR)) < 0 )
    	continue;
    	
    	if( ioctl( fd, IOW_GETINFO, &info))
				perror( "Unable to retrieve device info" );
		else {
			if ( strncmp(serial, (char*)info.serial, 8) == 0 && info.if_num != 0)
				return fd;
		}
    	close(fd);
  	}	
	return -1;
}

int iow_i2c_write(char address, char *buf, int size, int report_size, int fd) {

	// all size values in databytes, without address byte!
	char send_buf[8];
	char rec_buf[8];
	int i,j;
	int total_bytes_written = 0;

	if (size == 0) return 0;
	
	// device address needs the first data byte, so that only 5 dbs are left in the first package
	int num_pack = (size / 6) + 1;
	unsigned char num_send_now = 0;

	send_buf[0] = 0x02;	// report id 2
	//send_buf[1] is set later according to message no (contains start/stop of iic mesg)
	send_buf[2] = address << 1 | 0x00; // Bit 0 is 0 for write

	for (i = 0; i < num_pack; i++) {
		if (num_pack == 1) {
			num_send_now = size;
			send_buf[1] = 0xc0 | (size+1); // start+stop and num databytes, address needs one more byte
			memcpy(&send_buf[3], buf, num_send_now);	// append after second byte
		}
		else if (num_pack > 1 && i == 0) {	// first of multiple messages
				num_send_now = 5;
				send_buf[1] = 0x80 | (num_send_now + 1); // start bit and 6 databytes
				memcpy(&send_buf[3], buf, num_send_now);	// append after second byte
			}
		else if (num_pack > 1 && i < num_pack - 1) { // moddle of multiple messages
				num_send_now = 6;
				send_buf[1] = 0x00 | num_send_now; // no start/stop bit and 6 databytes
				memcpy(&send_buf[2], buf, num_send_now);	// append after first byte, no need to send address now
			}
		else if (num_pack > 1 && i == num_pack - 1) {	// last one of multiple messages
				num_send_now = (size + 1) % 6;	
				send_buf[1] = 0x40 | num_send_now; // stop bit and rest of databytes
				memcpy(&send_buf[2], buf, num_send_now);	// append after first byte, no need to send address now		
			}		

		buf += num_send_now;

		if (write(fd, send_buf, report_size) != report_size)
			return -1;

		if (read(fd, rec_buf, report_size) != report_size)
			return -2;
		
		if (rec_buf[1] >> 7) 
			return -3;
		
		total_bytes_written += rec_buf[1];
	}
	return total_bytes_written - 1; // address bytes do not count, return only num dbs
}

int iow_i2c_read(char address, char *buf, int size, int report_size, int fd) {

	char send_buf[8];
	char rec_buf[8];
	int i,j;
	int bytes_read = 0;

	if (size == 0) return 0;
	
	int num_packages_rec = ((size-1) / 6) + 1;
	
	send_buf[0] = 0x03;	// report id 3
	send_buf[1] = size; // num databytes t read
	send_buf[2] = address << 1 | 0x01; // Bit 0 is 1 for read
	
	// write report id 3
	if (write(fd, send_buf, report_size) != report_size)
		return -1;

	for (i=0; i < num_packages_rec; i++){
		// read multiple reports 
		if (read(fd, rec_buf, report_size) != report_size)
			return -1;

		if (rec_buf[1] >> 7)
			return -1;
		
		memcpy(&buf[bytes_read], &rec_buf[2], rec_buf[1]);
		bytes_read += rec_buf[1];		
	}
	return bytes_read;
}

int iow_i2c_init(int fd) {
		
	struct iowarrior_info info;
	char buf[2];
	int i;

	// get device info
	if( ioctl(fd, IOW_GETINFO, &info))
		printf("iow_init(): Unable to retrieve device info" );
		
	// enable iic function
	buf[0] = 0x01;	// report ID 1
	buf[1] = 0x01;	// enable iic
		
	if (write(fd, buf, info.report_size) != info.report_size)
		return -1;

	return info.report_size;
}



